# coding: utf-8
import logging
from collections import defaultdict
from typing import Dict, List

import arrow
import more_itertools
import yenv
from django.db.models import Q
from django.conf import settings
from cia_stuff.utils import iterables
from django.db import transaction
from django.db.models import Q, F

from review.lib import datetimes
from review.core import const, models
from review.core import models as core_models
from review.staff import logic
from review.staff import models as staff_models

logger = logging.getLogger(__name__)


@transaction.atomic
def denormalize_person_review_roles(
    skip_calibration=False,
    person_review_ids: List[int]=None,
    review_id: int=None,
    person_ids: List[int]=None,
    calibration_id: int=None,
    **kwargs
):
    person_review_filter = _get_person_review_filter(
        kwargs,
        person_review_ids=person_review_ids,
        review_id=review_id,
        person_ids=person_ids,
        calibration_id=calibration_id,
    )
    logger.info('Started denormalizing by filter: {}'.format(person_review_filter))
    if skip_calibration:
        created, updated = 0, 0
    else:
        created, updated = _denormalize_calibration_roles(**person_review_filter)
    created += _denormalize_review_roles(**person_review_filter)
    deleted_inh, created_inh = _denormalize_inherited_roles(**person_review_filter)
    logger.info('Finished denormalizing by filter: {}'.format(person_review_filter))
    return dict(
        deleted=deleted_inh,
        created=created_inh + created,
        updated=updated,
        filter=person_review_filter,
    )


def _get_person_review_filter(
    init_filter,
    person_review_ids: List[int]=None,
    review_id: int=None,
    person_ids: List[int]=None,
    calibration_id: int=None,
) -> Dict:
    res = dict(init_filter)
    if person_review_ids is not None:
        res['id__in'] = person_review_ids
    if review_id is not None:
        res['review_id'] = review_id
    if person_ids is not None:
        res['person_id__in'] = person_ids
    if calibration_id is not None:
        res['calibration_person_review__calibration_id'] = calibration_id
    return res


def async_denormalize_person_review_roles(
    skip_calibration=True,
    person_review_ids: List[int]=None,
    review_id: int=None,
    person_ids: List[int]=None,
    calibration_id: int=None,
):
    from review.core import tasks
    task = tasks.denormalize_person_review_roles_task
    if yenv.type != 'development':
        task = task.delay
    task(
        skip_calibration=skip_calibration,
        person_review_ids=person_review_ids,
        review_id=review_id,
        person_ids=person_ids,
        calibration_id=calibration_id,
    )


def async_denormalyze_calibration_roles(**kwargs):
    person_review_filter = _get_person_review_filter({}, **kwargs)
    _denormalize_calibration_roles(**person_review_filter)
    async_denormalize_person_review_roles(skip_calibration=True, **kwargs)


def _denormalize_calibration_roles(**person_review_filter):
    not_draft = ~Q(calibration__status=const.CALIBRATION_STATUS.DRAFT)
    calibrators = not_draft & Q(type=const.ROLE.CALIBRATION.CALIBRATOR)
    admins = Q(type=const.ROLE.CALIBRATION.ADMIN)
    not_empty_calibration = Q(
        calibration__calibration_person_review__isnull=False,
    )
    query_filter = (admins | calibrators) & not_empty_calibration
    if person_review_filter:
        query_filter &= Q(**{
            'calibration__calibration_person_review__person_review__' + key: value
            for key, value in person_review_filter.items()
        })
    info_to_update = models.CalibrationRole.objects.filter(query_filter).values_list(
        'id',
        'type',
        'calibration__calibration_person_review__person_review_id',
        'person_id',
    )
    denormalized_roles = set(models.PersonReviewRole.objects.filter(
        type__in=(
            const.ROLE.PERSON_REVIEW.CALIBRATION_ADMIN_DENORMALIZED,
            const.ROLE.PERSON_REVIEW.CALIBRATOR_ARCHIVED,
            const.ROLE.PERSON_REVIEW.CALIBRATOR_DENORMALIZED,
        ),
    ).values_list(
        'person_review_id',
        'from_calibration_role_id',
    ))
    created = models.PersonReviewRole.objects.bulk_create(
        models.PersonReviewRole(
            from_calibration_role_id=id_,
            type=const.ROLE.DENORMALIZATION[type_],
            person_review_id=person_review_id,
            person_id=person_id,
        )
        for id_, type_, person_review_id, person_id in info_to_update
        if (person_review_id, id_) not in denormalized_roles
    )
    updated = models.PersonReviewRole.objects.filter(
        type=const.ROLE.PERSON_REVIEW.CALIBRATOR_DENORMALIZED,
        from_calibration_role__calibration__status=const.CALIBRATION_STATUS.ARCHIVE,
    ).update(type=const.ROLE.PERSON_REVIEW.CALIBRATOR_ARCHIVED)
    updated += models.PersonReviewRole.objects.filter(
        type=const.ROLE.PERSON_REVIEW.CALIBRATOR_ARCHIVED,
        from_calibration_role__calibration__status=const.CALIBRATION_STATUS.IN_PROGRESS,
    ).update(type=const.ROLE.PERSON_REVIEW.CALIBRATOR_DENORMALIZED)
    return len(created), updated


def _denormalize_review_roles(**person_review_filter):
    filter_q = {}
    if person_review_filter:
        filter_q = {
            'person_review__' + key: value
            for key, value in person_review_filter.items()
        }

    roles_data = models.Review.objects.filter(
        ~Q(status=const.REVIEW_STATUS.DRAFT),
        role__type=const.ROLE.REVIEW.SUPERREVIEWER,
        **filter_q
    ).values(
        'role__id',
        'role__person__id',
        'person_review__id',
    )
    roles_should_be = {
        (datum['person_review__id'], datum['role__id']): datum['role__person__id']
        for datum in roles_data
        if datum['person_review__id'] is not None
    }

    roles_exist = set(models.PersonReviewRole.objects.filter(
        type=const.ROLE.PERSON_REVIEW.SUPERREVIEWER_DENORMALIZED,
    ).values_list(
        'person_review_id',
        'from_review_role_id',
    ))

    created = models.PersonReviewRole.objects.bulk_create(
        models.PersonReviewRole(
            person_id=person_id,
            person_review_id=person_review_id,
            from_review_role_id=review_role_id,
            type=const.ROLE.PERSON_REVIEW.SUPERREVIEWER_DENORMALIZED,
        )
        for (person_review_id, review_role_id), person_id
        in roles_should_be.items()
        if (person_review_id, review_role_id) not in roles_exist
    )
    return len(created)


def _denormalize_inherited_roles(**person_review_filter):
    person_ids = _get_affected_persons(**person_review_filter)
    deleted, created = 0, 0
    for cur_ids in more_itertools.chunked(person_ids, settings.CHUNK_FOR_DENORMALIZATION):
        inheritable_roles = _get_current_inheritable_roles(cur_ids)
        person_id_to_person_reviews, person_review_ids = _get_person_review_history(cur_ids)
        current_roles = _get_current_inherited_roles(person_review_ids)
        correct_roles = _get_correct_inherited_roles(inheritable_roles, person_id_to_person_reviews)
        to_delete, to_create = _create_models_update(current_roles, correct_roles)
        _update_models(to_delete, to_create)
        deleted += len(to_delete)
        created += len(to_create)
    return deleted, created


def _get_affected_persons(**person_review_filter):
    person_reviews = models.PersonReview.objects.filter(**person_review_filter)
    return set(person_reviews.values_list('person_id', flat=True))


def _get_current_inheritable_roles(person_ids):
    queryset = core_models.PersonReviewRole.objects.filter(
        ~Q(person_review__review__status=const.REVIEW_STATUS.DRAFT),
        type__in=const.ROLE.INHERITABLE_WITH_DENORMALIZATION,
        person_review__person_id__in=person_ids
    ).values(
        'id',
        'person_id',
        'type',
        'person_review__person_id',
        'person_review__review__finish_date',
    )
    grouped_by_type = defaultdict(list)
    for item in queryset:
        role_type = item.pop('type')
        grouped_by_type[role_type].append(item)
    return grouped_by_type


def _get_person_review_history(person_ids):
    person_reviews_history = core_models.PersonReview.objects.filter(
        person_id__in=person_ids
    ).values(
        'id',
        'person_id',
        'review__finish_date',
        'review__type',
    )
    by_person_id = defaultdict(list)
    for history in person_reviews_history:
        person_id = history.pop('person_id')
        by_person_id[person_id].append(history)
    by_person_id = {
        id_: iterables.DateRangeable(history, date_field='review__finish_date')
        for id_, history in by_person_id.items()
    }
    ids = [r['id'] for r in person_reviews_history]
    return by_person_id, ids


def _get_current_inherited_roles(person_reviews):
    return {
        tuple(item[:-1]): item[-1] for item in
        core_models.PersonReviewRole.objects.filter(
            type__in=const.ROLE.INHERITED,
            person_review_id__in=person_reviews
        ).values_list(
            'person_id',
            'person_review_id',
            'type',
            'inherited_from_id',
            'id',
        )
    }


def _get_correct_inherited_roles(grouped_by_type, grouped_by_person):
    correct_roles = set()
    for role_type, roles in grouped_by_type.items():
        for role in roles:
            owner_id = role['person_id']
            subject_id = role['person_review__person_id']
            role_id = role['id']
            review_finish_date = role['person_review__review__finish_date']
            person_history = grouped_by_person[subject_id]
            if role_type in (
                const.ROLE.PERSON_REVIEW.REVIEWER,
                const.ROLE.PERSON_REVIEW.TOP_REVIEWER,
                const.ROLE.PERSON_REVIEW.SUPERREVIEWER_DENORMALIZED,
            ):
                history_piece = person_history.get_range(to_date=review_finish_date)
            elif role_type in (
                const.ROLE.PERSON_REVIEW.READER,
                const.ROLE.PERSON_REVIEW.SUPERREADER,
                const.ROLE.PERSON_REVIEW.CALIBRATOR_DENORMALIZED,
                const.ROLE.PERSON_REVIEW.CALIBRATION_ADMIN_DENORMALIZED,
            ):
                history = list(reversed(person_history.get_range(
                    to_date=datetimes.shifted(review_finish_date, days=-1),
                )))
                history_piece = []
                if history and history[0]['review__type'] == const.REVIEW_TYPE.MIDDLE:
                    history_piece.append(history.pop(0))
                history_piece += [
                    pr for pr in history
                    if pr['review__type'] != const.REVIEW_TYPE.MIDDLE
                ][:const.CALIBRATION_INHERITANCE_LENGTH]
            else:
                history_piece = []
            for history_person_review in reversed(history_piece):
                correct_roles.add((
                    owner_id,
                    history_person_review['id'],
                    role_type if role_type in const.ROLE.INHERITED
                    else const.ROLE.INHERITANCE_WITH_DENORMALIZATION[role_type],
                    role_id,
                ))
    return correct_roles


def _create_models_update(current_roles, correct_roles):
    to_delete_keys = set(current_roles.keys()) - correct_roles
    to_delete = [current_roles[key] for key in to_delete_keys]
    to_create = [
        core_models.PersonReviewRole(
            person_id=person_id,
            person_review_id=person_review_id,
            type=role_type,
            inherited_from_id=role_id,
        )
        for (
            person_id,
            person_review_id,
            role_type,
            role_id,
        ) in correct_roles - set(current_roles.keys())
    ]
    logger.info('Update person review role inheritance: delete {} roles, create {} roles'.format(
        len(to_delete), len(to_create)
    ))
    return to_delete, to_create


def _update_models(to_delete, to_create):
    core_models.PersonReviewRole.objects.filter(
        id__in=to_delete
    ).delete()
    core_models.PersonReviewRole.objects.bulk_create(to_create)


def has_global_roles(person, global_roles):
    return get_existing_global_roles(person, global_roles).exists()


def get_existing_global_roles(person, roles):
    person = logic.ensure_person_model(person)
    return models.GlobalRole.objects.filter(
        person=person,
        type__in=roles,
    ).values_list('type', flat=True)
