# coding: utf-8
import uuid

from django.core.exceptions import ValidationError
from django.db import models
import jsonfield


from review.core import const
from review.lib.s3 import S3Storage


class Review(models.Model):
    name = models.CharField(max_length=255)

    start_date = models.DateField(db_index=True)
    finish_date = models.DateField(db_index=True)

    type = models.CharField(
        choices=const.REVIEW_TYPE.CHOICES,
        default=const.REVIEW_TYPE.NORMAL,
        max_length=1,
    )
    bonus_type = models.CharField(
        choices=const.REVIEW_BONUS_TYPE.CHOICES,
        default=const.REVIEW_BONUS_TYPE.HALFYEARLY,
        max_length=1,
    )
    bonus_reason = models.TextField(null=True)

    finish_feedback_date = models.DateField(
        null=True,
        default=None,
        blank=True,
    )
    finish_submission_date = models.DateField(
        null=True,
        default=None,
        blank=True,
    )
    finish_calibration_date = models.DateField(
        null=True,
        default=None,
        blank=True,
    )
    finish_approval_date = models.DateField(
        null=True,
        default=None,
        blank=True,
    )
    evaluation_from_date = models.DateField(
        null=True,
        default=None,
        blank=True,
    )
    evaluation_to_date = models.DateField(
        null=True,
        default=None,
        blank=True,
    )
    feedback_from_date = models.DateField(
        null=True,
        default=None,
        blank=True,
    )
    feedback_to_date = models.DateField(
        null=True,
        default=None,
        blank=True,
    )
    include_dismissed_after_date = models.DateField(
        null=True,
        default=None,
        blank=True,
    )

    status = models.CharField(
        choices=const.REVIEW_STATUS.CHOICES,
        default=const.REVIEW_STATUS.DRAFT,
        db_index=True,
        max_length=1,
    )

    scale = models.ForeignKey(
        to='core.MarksScale',
        related_name='reviews',
        related_query_name='review',
        on_delete=models.PROTECT,
        null=True,
    )
    mark_mode = models.CharField(
        choices=const.REVIEW_MODE.CHOICES_FOR_MARK,
        default=const.REVIEW_MODE.MODE_DISABLED,
        max_length=1,
    )
    goldstar_mode = models.CharField(
        choices=const.REVIEW_MODE.CHOICES_FOR_GOLDSTAR,
        default=const.REVIEW_MODE.MODE_DISABLED,
        max_length=1,
    )
    level_change_mode = models.CharField(
        choices=const.REVIEW_MODE.CHOICES_FOR_LEVEL_CHANGE,
        default=const.REVIEW_MODE.MODE_DISABLED,
        max_length=1,
    )
    salary_change_mode = models.CharField(
        choices=const.REVIEW_MODE.CHOICES_FOR_GOODIE,
        default=const.REVIEW_MODE.MODE_DISABLED,
        max_length=1,
    )
    bonus_mode = models.CharField(
        choices=const.REVIEW_MODE.CHOICES_FOR_GOODIE,
        default=const.REVIEW_MODE.MODE_DISABLED,
        max_length=1,
    )
    options_rsu_mode = models.CharField(
        choices=const.REVIEW_MODE.CHOICES_FOR_GOODIE,
        default=const.REVIEW_MODE.MODE_DISABLED,
        max_length=1,
    )
    options_rsu_format = models.CharField(
        choices=const.REVIEW_OPTIONS_RSU_FORMAT.CHOICES,
        default=const.REVIEW_OPTIONS_RSU_FORMAT.ACTUAL,
        max_length=1,
    )
    options_rsu_unit = models.CharField(
        choices=const.REVIEW_OPTIONS_RSU_UNIT.CHOICES,
        default=const.REVIEW_OPTIONS_RSU_UNIT.QUANTITY,
        max_length=1,
    )
    deferred_payment_mode = models.CharField(
        choices=const.REVIEW_MODE.CHOICES_FOR_GOODIE,
        default=const.REVIEW_MODE.MODE_DISABLED,
        max_length=1,
        null=True,
    )

    # comma separated days of week
    notify_reminder_days = models.CharField(
        max_length=255,
        blank=True,
        default='',
    )
    notify_reminder_date_from = models.DateField(
        null=True,
        blank=True,
    )
    notify_reminder_date_to = models.DateField(
        blank=True,
        null=True,
    )
    notify_events_other = models.BooleanField(default=False)
    notify_events_superreviewer = models.BooleanField(default=False)

    author = models.ForeignKey(
        to='staff.Person',
        related_name='created_reviews',
        related_query_name='created_review',
        on_delete=models.PROTECT,
        null=True,
    )

    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    salary_date = models.DateField(null=True, blank=True)

    kpi_loaded = models.DateTimeField(null=True)
    product_schema_loaded = models.DateTimeField(null=True, blank=True)

    # мигрировано из старой ревьюшницы
    migrated = models.BooleanField(default=False)

    class Meta:
        index_together = ('id', 'status')

    def __str__(self):
        return '%s %s' % (self.id, self.name)

    @staticmethod
    def get_real_fields():
        return {
            f.name for f in Review._meta.get_fields() if not f.is_relation
        } | {'author'}


def _validate_scale(scale):
    if any(not isinstance(value, (int, float, dict)) for value in list(scale.values())):
        raise ValidationError('Wrong scale. Values have to be int or float or dict')


class MarksScale(models.Model):
    scale = jsonfield.JSONField(validators=[_validate_scale])
    show_absolute = models.BooleanField(
        default=False,
    )
    use_colors = models.BooleanField(
        default=True,
    )
    version = models.IntegerField(
        default=2,
    )


class Goodie(models.Model):
    review = models.ForeignKey('core.Review', related_name='goodies')
    level = models.SmallIntegerField(default=0)
    mark = models.CharField(
        max_length=16,
        default=const.MARK.DEFAULT,
    )
    goldstar = models.CharField(
        max_length=1,
        choices=const.GOLDSTAR.CHOICES,
        default=const.GOLDSTAR.DEFAULT,
    )
    level_change = models.SmallIntegerField(
        default=const.REVIEW_MODE.DEFAULTS[const.FIELDS.LEVEL_CHANGE],
    )
    salary_change = models.DecimalField(
        max_digits=7,
        decimal_places=2,
        default=0,
    )
    bonus = models.DecimalField(
        max_digits=7,
        decimal_places=2,
        default=0,
    )
    options_rsu = models.IntegerField(
        default=0,
    )

    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    class Meta:
        unique_together = (
            'review',
            'level',
            'mark',
            'goldstar',
            'level_change',
        )


class PersonReview(models.Model):
    # по многим полям можно сделать индекс, но пока предполагается, что
    # эти сущности будут фильтроваться в памяти по множеству фильтров, а в
    # базе по парочке типа person_id, review_id

    person = models.ForeignKey(
        to='staff.Person',
        related_name='person_reviews',
        related_query_name='person_review',
    )
    review = models.ForeignKey(
        to='core.Review',
        related_name='person_reviews',
        related_query_name='person_review',
    )

    status = models.CharField(
        max_length=1,
        choices=const.PERSON_REVIEW_STATUS.CHOICES,
        default=const.PERSON_REVIEW_STATUS.WAIT_EVALUATION,
        db_index=True,
    )
    # это поле в принципе работает только в статусе evaluation,
    # в остальных статусах логично делать его NULL
    approve_level = models.IntegerField(
        default=0,
    )

    # по умолчанию MARK.NOT_SET — значит оценку еще не ставили
    # прочерк значит, что явно поставили, что оценка не нужна
    # хотя можно было бы исключать людей из ревью (пока вроде как решили, что
    # NO_MARK оставляем, но рекомендуем выкидывать из ревью)
    mark = models.CharField(
        max_length=16,
        default=const.MARK.NOT_SET,
    )
    # голдстар — это модификатор оценки.
    # В зависимости от комбинации уровня, оценки и голдстара из файла плюшек
    # берутся рекомендуемые значения повышений/премий/опционов
    goldstar = models.CharField(
        max_length=1,
        choices=const.GOLDSTAR.CHOICES,
        default=const.GOLDSTAR.NO,
    )
    level_change = models.SmallIntegerField(
        default=0,
    )
    # percent
    salary_change = models.DecimalField(
        max_digits=7,
        decimal_places=2,
        default=0,
    )
    salary_change_absolute = models.DecimalField(
        max_digits=20,
        decimal_places=2,
        default=0,
    )
    salary_change_type = models.CharField(
        max_length=1,
        choices=const.SALARY_DEPENDENCY_TYPE.CHOICES,
        default=const.SALARY_DEPENDENCY_TYPE.DEFAULT,
    )
    # percent
    bonus = models.DecimalField(
        max_digits=7,
        decimal_places=2,
        default=0,
    )
    bonus_absolute = models.DecimalField(
        max_digits=20,
        decimal_places=2,
        default=0,
    )
    bonus_rsu = models.IntegerField(
        default=0,
    )
    bonus_type = models.CharField(
        max_length=1,
        choices=const.SALARY_DEPENDENCY_TYPE.CHOICES,
        default=const.SALARY_DEPENDENCY_TYPE.DEFAULT,
    )
    deferred_payment = models.DecimalField(
        max_digits=20,
        decimal_places=2,
        default=0,
        null=True,
    )
    options_rsu = models.IntegerField(
        default=0,
    )
    options_rsu_legacy = models.BooleanField(
        default=False,
    )

    flagged = models.BooleanField(
        default=False,
        db_index=True,
    )
    flagged_positive = models.BooleanField(
        default=False,
        db_index=True,
    )

    tag_average_mark = models.CharField(max_length=255, null=True)

    # признак "учитывается в среднюю" для аналитиков
    taken_in_average = models.BooleanField(default=False)

    main_product = models.ForeignKey(
        to='gradient.MainProduct',
        null=True,
        blank=True,
        related_name='person_reviews',
        related_query_name='person_review',
    )
    umbrella = models.ForeignKey(
        to='gradient.Umbrella',
        null=True,
        blank=True,
        related_name='person_reviews',
        related_query_name='person_review',
    )

    updated_at = models.DateTimeField(db_index=True)
    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    class Meta:
        unique_together = ('person', 'review')
        indexes = (
            models.Index(fields=['tag_average_mark']),
        )

    def __str__(self):
        return '%s @ %s' % (self.person, self.review)


class PersonReviewChange(models.Model):
    subject = models.ForeignKey(
        to='staff.Person',
        related_name='changes',
        related_query_name='change',
    )
    subject_type = models.CharField(
        choices=const.PERSON_REVIEW_CHANGE_TYPE.CHOICE,
        default=const.PERSON_REVIEW_CHANGE_TYPE.PERSON,
        db_index=True,
        max_length=10,
    )
    person_review = models.ForeignKey(
        to='core.PersonReview',
        related_name='changes',
        related_query_name='change',
    )
    diff = jsonfield.JSONField()

    created_at = models.DateTimeField(db_index=True)
    created_at_auto = models.DateTimeField(auto_now_add=True)

    notified = models.BooleanField(db_index=True, default=False)

    class Meta:
        indexes = [
            # for log
            models.Index(fields=['person_review', 'created_at']),
            # for notifications
            models.Index(fields=['created_at', 'notified']),
        ]

    def __str__(self):
        return '%s: %s' % (self.person_review_id, self.diff)


class PersonReviewComment(models.Model):
    subject = models.ForeignKey(
        to='staff.Person',
        related_name='comments',
        related_query_name='comment',
    )
    subject_type = models.CharField(
        choices=const.PERSON_REVIEW_CHANGE_TYPE.CHOICE,
        default=const.PERSON_REVIEW_CHANGE_TYPE.PERSON,
        db_index=True,
        max_length=10,
    )
    person_review = models.ForeignKey(
        to='core.PersonReview',
        related_name='comments',
        related_query_name='comment',
    )
    text_wiki = models.TextField()

    created_at = models.DateTimeField(db_index=True)
    updated_at = models.DateTimeField(db_index=True)

    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    notified = models.BooleanField(db_index=True, default=False)

    class Meta:
        indexes = [
            # for log
            models.Index(fields=['person_review', 'created_at']),
            # for notifications
            models.Index(fields=['created_at', 'notified']),
        ]

    def __str__(self):
        return '%s: %s' % (self.person_review_id, self.text_wiki[:100])


class Calibration(models.Model):
    name = models.CharField(max_length=255, null=True)

    status = models.CharField(
        choices=const.CALIBRATION_STATUS.CHOICES,
        default=const.CALIBRATION_STATUS.DRAFT,
        db_index=True,
        max_length=1,
    )

    start_date = models.DateField(db_index=True, null=True)
    finish_date = models.DateField(db_index=True, null=True)

    author = models.ForeignKey(
        to='staff.Person',
        related_name='created_calibrations',
        related_query_name='created_calibration',
        on_delete=models.PROTECT,
        null=True,
    )

    notify_users = models.BooleanField(default=True)

    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    def __str__(self):
        return '%s %s' % (self.id, self.name)


class CalibrationPersonReview(models.Model):
    calibration = models.ForeignKey(
        to='core.Calibration',
        related_name='calibration_person_reviews',
        related_query_name='calibration_person_review',
    )
    person_review = models.ForeignKey(
        to='core.PersonReview',
        related_name='calibration_person_reviews',
        related_query_name='calibration_person_review',
    )

    discuss = models.BooleanField(
        default=False,
        db_index=True,
    )

    updated_at = models.DateTimeField(db_index=True)
    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    class Meta(object):
        unique_together = ('calibration', 'person_review')

    def __str__(self):
        return '%s @ %s' % (self.person_review, self.calibration)


class GlobalRole(models.Model):
    person = models.ForeignKey(
        to='staff.Person',
        related_name='global_roles',
        related_query_name='global_role',
    )

    type = models.CharField(
        max_length=2,
        choices=const.ROLE.GLOBAL.CHOICES,
        db_index=True,
    )

    class Meta:
        indexes = [
            models.Index(fields=['person', 'type'])
        ]
        unique_together = ('person', 'type')

    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)


class ReviewRole(models.Model):
    person = models.ForeignKey(
        to='staff.Person',
        related_name='review_roles',
        related_query_name='review_role',
    )
    review = models.ForeignKey(
        to='core.Review',
        related_name='roles',
        related_query_name='role',
    )
    type = models.CharField(
        max_length=2,
        choices=const.ROLE.REVIEW.CHOICES,
        db_index=True,
    )

    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    class Meta(object):
        unique_together = ('person', 'review', 'type')
        # этот индекс может использоваться для получения ролей по
        # сотруднику и типу роли
        index_together = ('person', 'type')

    def __str__(self):
        return '%s for %s @ %s' % (self.type, self.person, self.review)


class PersonReviewRole(models.Model):
    person = models.ForeignKey(
        to='staff.Person',
        related_name='person_review_roles',
        related_query_name='person_review_role'
    )
    person_review = models.ForeignKey(
        to='core.PersonReview',
        related_name='roles',
        related_query_name='role',
        on_delete=models.CASCADE
    )
    type = models.CharField(
        max_length=2,
        choices=const.ROLE.PERSON_REVIEW.CHOICES,
        db_index=True,
    )

    position = models.SmallIntegerField(default=0)

    # некоторые из ролей подразумевают доступ ко всем предыдущим или
    # к двум предыдущим PersonReview. Долго подумав над тем как это
    # селектить в коде, чтобы это было эффективно и не получился
    # сверхзапутанный код, я решил, что проще будет поддерживать
    # денормализованное наследование ролей — тогда мы не потеряем
    # в производительности и код будет понятным.
    inherited_from = models.ForeignKey(
        to='self',
        null=True,
        on_delete=models.CASCADE,
    )
    from_calibration_role = models.ForeignKey(
        to='core.CalibrationRole',
        null=True,
        on_delete=models.CASCADE,
    )
    from_review_role = models.ForeignKey(
        to='core.ReviewRole',
        null=True,
        on_delete=models.CASCADE,
    )

    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    class Meta(object):
        unique_together = (
            'person',
            'person_review',
            'type',
            'position',
            'inherited_from',
        )
        # этот индекс может использоваться для получения ролей по
        # сотруднику и типу роли
        index_together = ('person', 'type')

    def __str__(self):
        return '%s for %s in %s' % (self.type, self.person, self.person_review)


class CalibrationRole(models.Model):
    person = models.ForeignKey(
        to='staff.Person',
        related_name='calibration_roles',
        related_query_name='calibration_role',
    )
    calibration = models.ForeignKey(
        to='core.Calibration',
        related_name='roles',
        related_query_name='role'
    )
    type = models.CharField(
        max_length=2,
        choices=const.ROLE.CALIBRATION.CHOICES,
        db_index=True,
    )
    notified = models.BooleanField(
        default=False,
    )

    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    class Meta(object):
        unique_together = ('person', 'calibration', 'type')
        index_together = (
            ('person', 'type'),
            ('type', 'notified'),
        )

    def __str__(self):
        return '%s for %s @ %s' % (self.type, self.person, self.calibration)


class StoredFilter(models.Model):
    owner = models.ForeignKey(
        to='staff.Person',
        related_name='review_filters+',
        related_query_name='review_filters+',
    )
    name = models.CharField(max_length=255)
    value = jsonfield.JSONField()

    def __str__(self):
        return '%s@ %s' % (self.owner.login, self.name)


def _get_upload_filename(instance, filename):
    prefix = ''
    if hasattr(instance, 'review_id'):
        prefix = '{}_'.format(instance.review_id)

    return '{prefix}{filename}'.format(
        prefix=prefix,

        # ignore provided filename because s3 paths should not contain russian symbols
        filename=uuid.uuid4().hex
    )


class ExcelTemplate(models.Model):
    review = models.ForeignKey(
        to='core.Review',
        related_name='excel_templates',
        related_query_name='excel_template',
    )
    file = models.FileField(
        upload_to=_get_upload_filename,
        max_length=512,
        storage=S3Storage(),
        null=True,
        blank=True,
    )
    template_type = models.CharField(
        choices=const.EXCEL_TEMPLATE_TYPE.CHOICES,
        default=const.EXCEL_TEMPLATE_TYPE.SECRET,
        max_length=255,
    )

    class Meta:
        unique_together = ('review', 'template_type')

    def __str__(self):
        return '%s, %s' % (self.review, self.template_type)


class Kpi(models.Model):
    person_review = models.ForeignKey(
        to=PersonReview,
        related_name='kpis',
        related_query_name='kpi',
    )

    name = models.CharField(
        max_length=255,
    )

    year = models.SmallIntegerField(null=True)
    quarter = models.SmallIntegerField(null=True)
    weight = models.IntegerField(null=True)

    percent = models.SmallIntegerField()
