import re

from review.core import const as core_const
from review.lib.serializers import BaseSerializer, F
from review.staff.serializers import PersonSerializer


class ReviewSerializer(BaseSerializer):
    log_instance_name = 'Review'

    REVIEW_MODE = core_const.REVIEW_MODE
    MARK_MODE = {
        REVIEW_MODE.FORM_MARK_MODE_DISABLED: (REVIEW_MODE.MODE_DISABLED, REVIEW_MODE.MODE_DISABLED),
        REVIEW_MODE.FORM_MARK_MODE_GOLDSTAR_DISABLED: (REVIEW_MODE.MODE_MANUAL, REVIEW_MODE.MODE_DISABLED),
        REVIEW_MODE.FORM_MARK_MODE_GOLDSTAR_ENABLED: (REVIEW_MODE.MODE_MANUAL, REVIEW_MODE.MODE_MANUAL),
        REVIEW_MODE.FORM_MARK_MODE_GOLDSTAR_RESTRICTED: (REVIEW_MODE.MODE_MANUAL, REVIEW_MODE.MODE_MANUAL_BY_CHOSEN),
    }

    @classmethod
    def get_mark_mode(cls, review):
        inverse_mark_mode = {v: k for k, v in cls.MARK_MODE.items()}
        mark_mode = review.mark_mode
        goldstar_mode = review.goldstar_mode
        return inverse_mark_mode[(mark_mode, goldstar_mode)]

    @classmethod
    def get_notify_reminder_days(cls, review):
        notification_days = review.notify_reminder_days
        return [day for day in notification_days.split(',') if day]

    @classmethod
    def get_notify_reminder_type(cls, review):
        notify_days = review.notify_reminder_days
        notify_date_from = review.notify_reminder_date_from
        notify_date_to = review.notify_reminder_date_to

        if all((
                not notify_days,
                notify_date_from is None,
                notify_date_to is None,
        )):
            return core_const.REVIEW_NOTIFICATION_SETTINGS.NO
        elif all((
                notify_days,
                notify_date_from is not None,
                notify_date_to is not None,
        )):
            return core_const.REVIEW_NOTIFICATION_SETTINGS.PERIOD
        else:
            return core_const.REVIEW_NOTIFICATION_SETTINGS.ONCE

    fields = {
        'id',
        'name',

        'start_date',
        'finish_date',

        'scale_id',
        'finish_feedback_date',
        'finish_submission_date',
        'finish_calibration_date',
        'finish_approval_date',
        'include_dismissed_after_date',
        'evaluation_from_date',
        'evaluation_to_date',
        'feedback_from_date',
        'feedback_to_date',
        'bonus_reason',

        F('type', verbose=core_const.REVIEW_TYPE.VERBOSE),
        F('bonus_type', verbose=core_const.REVIEW_BONUS_TYPE.VERBOSE),
        F('status', verbose=core_const.REVIEW_STATUS.VERBOSE),

        F('mark_mode', source='get_mark_mode()'),
        F('level_change_mode', verbose=core_const.REVIEW_MODE.VERBOSE),
        F('salary_change_mode', verbose=core_const.REVIEW_MODE.VERBOSE),
        F('bonus_mode', verbose=core_const.REVIEW_MODE.VERBOSE),
        F('options_rsu_mode', verbose=core_const.REVIEW_MODE.VERBOSE),
        F('options_rsu_unit', verbose=core_const.REVIEW_OPTIONS_RSU_UNIT.VERBOSE),
        F('goldstar_mode', verbose=core_const.REVIEW_MODE.VERBOSE),
        F('deferred_payment_mode', verbose=core_const.REVIEW_MODE.VERBOSE),

        F(
            'options_rsu_format',
            verbose=core_const.REVIEW_OPTIONS_RSU_FORMAT.VERBOSE,
        ),

        F('notify_reminder_days', source='get_notify_reminder_days()'),
        F('notify_reminder_type', source='get_notify_reminder_type()'),
        'notify_reminder_date_from',
        'notify_reminder_date_to',
        'notify_events_other',
        'notify_events_superreviewer',
        F('author', complex=PersonSerializer)
    }

    default_fields = (
        'id',
        'name',
        'type',
        'start_date',
        'finish_date',
    )


class PersonReviewSerializer(BaseSerializer):
    log_instance_name = 'PersonReview'

    fields = {
        'id',
        'approve_level',
        'mark',
        'level_change',
        'bonus_rsu',
        'options_rsu',
        'options_rsu_legacy',
        'flagged',
        'flagged_positive',
        'tag_average_mark',
        'taken_in_average',

        F('status', verbose=core_const.PERSON_REVIEW_STATUS.VERBOSE),
        F('goldstar', verbose=core_const.GOLDSTAR.VERBOSE),
        F('bonus_type', verbose=core_const.SALARY_DEPENDENCY_TYPE.VERBOSE),
        F('salary_change_type', verbose=core_const.SALARY_DEPENDENCY_TYPE.VERBOSE),

        F('salary_change', cast_type=float),
        F('salary_change_absolute', cast_type=float),
        F('bonus', cast_type=float),
        F('bonus_absolute', cast_type=float),
        F('deferred_payment', cast_type=float),

        F('person', complex=PersonSerializer),
        F('review', complex=ReviewSerializer),
    }

    default_fields = (
        'review.id',
        'person.id',
        'id',
    )


class KpiSerializer(BaseSerializer):
    fields = {
        'name',
        F('q', source='get_q()'),
        F('year', source='get_year()'),
        'percent',
        'weight',
    }
    matcher = re.compile(r'\d{4}_Q\d$')

    @classmethod
    def get_q(cls, obj):
        return 'Q{}'.format(obj.quarter)

    @classmethod
    def get_year(cls, obj):
        return str(obj.year)


class CalibrationSerializer(BaseSerializer):
    fields = {
        'id',
        'name',
        'start_date',
        'finish_date',

        F('status', verbose=core_const.CALIBRATION_STATUS.VERBOSE),
    }
