# coding: utf-8
from review.core import const as core_const
from review.core import serializers_core
from review.lib.serializers import BaseSerializer, F
from review.staff.serializers import PersonSerializer
from review.staff import const as staff_const
from review.core.logic import legacy
from django.conf import settings
from decimal import Decimal
import logging
from collections import OrderedDict

PERSON_REVIEW_ROLES_VERBOSE = core_const.ROLE.PERSON_REVIEW.VERBOSE.copy()
PERSON_REVIEW_ROLES_VERBOSE.update({
    core_const.ROLE.DEPARTMENT.HEAD: 'head',
    core_const.ROLE.PERSON.SELF: 'self',
    core_const.ROLE.CALIBRATION.ADMIN: core_const.ROLE.CALIBRATION.VERBOSE[core_const.ROLE.CALIBRATION.ADMIN],
})

CALIBRATION_ROLES_VERBOSE = core_const.ROLE.CALIBRATION.VERBOSE.copy()
CALIBRATION_ROLES_VERBOSE.update({
    r: core_const.ROLE.GLOBAL.VERBOSE[r] for r in core_const.ROLE.CALIBRATION_GLOBAL_ROLES
})

CALIBRATION_PERSON_REVIEW_ROLES_VERBOSE = {
    role: core_const.ROLE.CALIBRATION.VERBOSE[role]
    for role in core_const.ROLE.CALIBRATION.ALL
}

logger = logging.getLogger(__name__)


class PersonReviewActionsSerializer(BaseSerializer):
    fields = {
        'tag_average_mark',
        'taken_in_average',
        'mark',
        'goldstar',
        'level_change',
        'salary_change',
        'salary_change_absolute',
        'bonus',
        'bonus_absolute',
        'bonus_rsu',
        'options_rsu',
        'flagged',
        'flagged_positive',
        'approve',
        'unapprove',
        'announce',
        'allow_announce',
        'comment',
        'reviewers',
        'umbrella',
        'main_product',
        'deferred_payment',
    }


class PersonReviewIsDifferFromDefaultSerializer(BaseSerializer):
    fields = {
        'level_change_is_differ_from_default',
        'salary_change_is_differ_from_default',
        'bonus_is_differ_from_default',
        'options_rsu_is_differ_from_default',
    }


class PersonReviewSalarySerializer(BaseSerializer):
    fields = {
        'value',
        'currency',
    }


class PersonReviewSalaryAfterReviewSerializer(BaseSerializer):
    log_instance_id_attr = 'value'
    fields = {
        'after_review',
        F('value', source='get_value()'),
        'currency',
    }

    @classmethod
    def get_value(cls, salary_dict):
        try:
            val = salary_dict.after_review
        except (KeyError, AttributeError):
            val = core_const.NO_ACCESS
        return val if isinstance(val, core_const.SPECIAL_VALUE) else float(val)


class ExtendedPersonSerializer(PersonSerializer):
    fields = PersonSerializer.combine_fields({
        'position',
        'first_name',
        'last_name',
        'department_id',
        'department_slug',
        'department_name',
        'department_path',
        'department_chain_names',
        'department_chain_slugs',
        'chief',
        'join_at',
        'city_name',
    })


class CalibratorPersonSerializer(PersonSerializer):
    fields = PersonSerializer.combine_fields({
        'chief',
    })


class ReviewRolesPersonsSerializer(ExtendedPersonSerializer):
    fields = {
        F(field, source='[{}]'.format(field))
        for field in core_const.FIELDS.DEFAULT_REVIEW_ROLE_PERSON_FIELDS - {'gender'}
    } | {
        F('gender', source='[gender]', verbose=staff_const.GENDER.VERBOSE)
    }


def verbose_list_serializer(_list, verbose_dict):
    return [verbose_dict[item] for item in _list]


class GoodieSerializer(serializers_core.BaseSerializer):
    fields = {
        'id',
        'level',
        'mark',
        F('goldstar', source='goldstar', verbose=core_const.GOLDSTAR.VERBOSE),
        'level_change',
        'salary_change',
        'bonus',
        'options_rsu',
    }


class ExtendedReviewSerializer(serializers_core.ReviewSerializer):
    fields = serializers_core.ReviewSerializer.combine_fields({
        'actions',
        'goals_from_date',
        'goals_to_date',
        'salary_date',
        'kpi_loaded',
        'product_schema_loaded',
        F('permissions', source='get_permissions()'),
        F('admins', complex=ReviewRolesPersonsSerializer, many=True),
        F('super_reviewers', complex=ReviewRolesPersonsSerializer, many=True),
        F('accompanying_hrs', complex=ReviewRolesPersonsSerializer, many=True),
        F('roles', source='roles', verbose=core_const.ROLE.VERBOSE, many=True),
        F('goodies', complex=GoodieSerializer, many=True),
    }) - {'author'}

    @classmethod
    def get_permissions(cls, review):
        return list(review.permissions)


class PersonReviewReviewersSerializer(ExtendedPersonSerializer):
    fields = {
        F(field, source='[{}]'.format(field))
        for field in core_const.FIELDS.DEFAULT_REVIEWER_FIELDS - {
            'position',
            'person_review_id',
            'gender'
        }
    } | {
        F('gender', source='[gender]', verbose=staff_const.GENDER.VERBOSE)
    }

    # ревьюеры бывают объектом или списком
    @classmethod
    def serialize(cls, obj, *args, **kwargs):
        if isinstance(obj, list):
            return cls.serialize_many(obj, *args, **kwargs)
        return super(PersonReviewReviewersSerializer, cls).serialize(
            obj, *args, **kwargs)


class PersonReviewExtendedSerializer(serializers_core.PersonReviewSerializer):
    fields = serializers_core.PersonReviewSerializer.combine_fields({
        'mark_level_history',
        'level',
        'action_at',
        'profession',
        'fte',
        F('is_differ_from_default', source='*', complex=PersonReviewIsDifferFromDefaultSerializer),
        F('actions', source='action_*', complex=PersonReviewActionsSerializer),
        F('person', source='person_*', complex=ExtendedPersonSerializer),
        F('review', source='review_*', complex=ExtendedReviewSerializer),
        F('salary', source='salary_*', complex=PersonReviewSalarySerializer),
        F('salary_after_review', source='salary_*', complex=PersonReviewSalaryAfterReviewSerializer),
        F('reviewers', complex=PersonReviewReviewersSerializer, many=True),
        F('roles', source='roles', verbose=core_const.ROLE.VERBOSE, many=True),
        'goals_url',
        'st_goals_url',
        'feedback_url',
        'umbrella',
        'main_product',
        'product_schema_loaded',
        F('updated_at', source='get_updated_at()'),
    })

    @classmethod
    def get_roles(cls, person_review):
        result = {}
        if person_review.permissions_read != core_const.NOT_SET:
            result.update(person_review.permissions_read)
        if person_review.permissions_write != core_const.NOT_SET:
            result.update(person_review.permissions_write)
        return result

    @classmethod
    def get_updated_at(cls, obj):
        if obj:
            return obj.updated_at.strftime('%Y-%m-%d %H:%M')

    default_fields = core_const.FIELDS.MINIMUM


class ExtendedCalibrationSerializer(serializers_core.CalibrationSerializer):
    fields = serializers_core.CalibrationSerializer.combine_fields({
        'actions',
        'author',
        'permissions',
        'review_ids',
        F('admins', complex=PersonSerializer, many=True),
        F('roles', source='roles', verbose=core_const.ROLE.VERBOSE, many=True),
    })


class CalibrationRolePersonSerializer(BaseSerializer):
    fields = {
        F('person', source='[person]', complex=CalibratorPersonSerializer),
        F('subordinates_total', source='[subordinates_total]'),
        F('subordinates_in_calibration', source='[subordinates_in_calibration]'),
    }


class CalibrationPersonReviewSerializer(BaseSerializer):
    fields = {
        'id',
        'discuss',
        'actions',
        'calibration_id',
        F('roles', source='roles', verbose=core_const.ROLE.VERBOSE, many=True),
        F('person_review', complex=PersonReviewExtendedSerializer)
    }

    @classmethod
    def serialize_with_person_review_fields(cls, obj, person_review_fields):
        return cls.serialize_many_with_person_review_fields([obj], person_review_fields)[0]

    @classmethod
    def serialize_many_with_person_review_fields(cls, objects, person_review_fields):
        requested_fields = ['person_review.' + field for field in person_review_fields]
        requested_fields += [
            field for field in cls.fields
            if not getattr(field, 'target', '') == 'person_review'
        ]
        return cls.serialize_many(objects, requested_fields)


class _NoSpecialValuesSerializer(PersonReviewExtendedSerializer):

    @classmethod
    def handle_special_value(cls, field_value):
        return ''

    @classmethod
    def get_many_ordered(cls, person_reviews):
        result = []
        for person_review in person_reviews:
            serialized = cls.serialize(person_review)

            serialized_with_order = OrderedDict()
            left = object()
            for field in cls.default_fields_ordered:
                value = serialized.get(field, left)
                if value is left:
                    logger.warning('%s %s' % (field, serialized))
                else:
                    serialized_with_order[field] = value

            result.append(serialized_with_order)
        return result


class CommentsSerializer(_NoSpecialValuesSerializer):

    default_fields_ordered = [
        'id',
        'login',
        'comments'
    ]
    default_fields = frozenset(default_fields_ordered)

    fields = PersonReviewExtendedSerializer.combine_fields({
        F('login', source='person_login'),
        F('comments', source='get_comments()'),

    })

    @classmethod
    def get_comments(cls, obj):
        sorted_cmnts = sorted(
            obj.comments,
            key=lambda it: (-it.created_at.toordinal(), it.id),
        )
        return '\n\n'.join(
            '{author} {date}:\n{comment}'.format(
                author=cmnt.subject_login,
                date=cmnt.created_at.strftime('%d/%m/%y'),
                comment=cmnt.text_wiki,
            )
            for cmnt in sorted_cmnts
        )


class PersonReviewsSerializer(_NoSpecialValuesSerializer):

    default_fields_ordered = [
        'id',
        'login',
        'full_name',
        'assignee',
        'tabled',
        'status',

        'scale',
        'grade_before_review',
        'grade_after_review',
        'grade_difference',

        'mark_at_review',
        'mark_at_review_as_number',
        'mark_at_previous_review',
        'mark_at_previous_review_as_number',
        'mark_before_the_previous_review',
        'mark_before_the_previous_review_as_number',

        'salary_before_review',
        'salary_after_review',
        'salary_change_percentage',

        'bonus',
        'bonus_payment_percentage',
        'bonus_option',
        'bonus_option_value',

        'extra_payment',
        'extra_payment_at_previous_review',
        'extra_option',
        'extra_option_at_previous_review',

        'currency',
        'fte',
        'full_department_name',
        'city',

        'review_link',
        'review_type',
        'feedback_link',
        'goals_link',

        'tag_average_mark',
        'salary_change_absolute',
        'bonus_rsu',

        'umbrella',
        'main_product',

        'taken_in_average',
        'updated_at',
        'deferred_payment',
    ]
    default_fields = frozenset(default_fields_ordered)

    fields = frozenset({
        F('full_name', source='get_full_name()'),
        F('login', source='person_login'),
        F('assignee', source='get_assignee()'),
        F('tabled', source='flagged'),
        F('scale', source='profession'),
        F('status', verbose=core_const.PERSON_REVIEW_STATUS.VERBOSE),

        F('grade_before_review', source='level'),
        F('grade_difference', source='level_change'),
        F('grade_after_review', source='get_grade_after_review()'),

        F('mark_at_review', source='get_mark_at_review()'),
        F('mark_at_review_as_number', source='get_mark_at_review_as_number()'),
        F(
            'mark_at_previous_review',
            source='get_mark_at_previous_review()'
        ),
        F(
            'mark_at_previous_review_as_number',
            source='get_mark_at_previous_review_as_number()'
        ),
        F(
            'mark_before_the_previous_review',
            source='get_mark_before_the_previous_review()'
        ),
        F(
            'mark_before_the_previous_review_as_number',
            source='get_mark_before_the_previous_review_as_number()'
        ),

        F('salary_before_review', source='salary_value'),
        F('salary_change_percentage', source='salary_change'),
        F('salary_after_review', source='salary_after_review'),
        F('salary_change_absolute', source='get_salary_change_absolute()'),

        F('bonus', source='bonus_absolute'),
        F('bonus_payment_percentage', source='bonus'),
        F('bonus_rsu', source='bonus_rsu'),
        F('bonus_option', source='get_bonus_option()'),
        F('bonus_option_value', source='options_rsu'),

        F('extra_payment', source='get_extra_payment()'),
        F(
            'extra_payment_at_previous_review',
            source='get_extra_payment_at_previous_review()'
        ),
        F('extra_option', source='get_extra_option()'),
        F(
            'extra_option_at_previous_review',
            source='get_extra_option_at_previous_review()',
        ),

        F('currency', source='salary_currency'),
        F('fte'),
        F('full_department_name', source='get_full_department_name()'),
        F('city', 'person_city_name'),

        F('review_link', source='get_review_link()'),
        F('review_type', verbose=core_const.REVIEW_TYPE.VERBOSE),
        F('feedback_link', source='get_feedback_link()'),
        F('goals_link', source='get_goals_link()'),

        F('main_product', source='get_main_product()'),
        F('umbrella', source='get_umbrella()'),

        F('updated_at', source='get_updated_at()'),
        F('deferred_payment', source='deferred_payment'),
    })

    fields = fields | (set(default_fields_ordered) - fields)

    @classmethod
    def get_full_name(cls, obj):
        return ' '.join([
            obj.person_last_name,
            obj.person_first_name,
        ])

    @classmethod
    def get_assignee(cls, obj):
        return ', '.join([person['login'] for person in obj.action_at])

    @classmethod
    def get_grade_after_review(cls, obj):
        if isinstance(obj.level, core_const.SPECIAL_VALUE):
            return obj.level
        if isinstance(obj.level_change, core_const.SPECIAL_VALUE):
            return obj.level_change
        if obj.level is None:
            return None
        return obj.level + obj.level_change

    @staticmethod
    def _mark_to_str(mark):
        if mark in (core_const.MARK.NOT_SET, None):
            return ''
        elif mark == '-':
            return core_const.MARK.TEXT_NO_MARK
        else:
            return mark

    @classmethod
    def _get_mark_from_history(cls, obj, pos):
        if len(obj.short_history) >= abs(pos):
            mark = obj.short_history[pos]['mark']
            scale = obj.short_history[pos]['scale'].get(mark)
            if isinstance(scale, dict):
                return scale['text_value']
            return cls._mark_to_str(mark)

    @classmethod
    def get_mark_at_previous_review(cls, obj):
        return cls._get_mark_from_history(obj, -1)

    @classmethod
    def cast_mark_to_num(cls, obj, mark):
        scale = obj.review_marks_scale.get(mark)
        if isinstance(scale, dict):
            scale = scale.get('value')
        return scale or core_const.MARK.TO_NUM.get(mark)

    @classmethod
    def get_mark_at_previous_review_as_number(cls, obj):
        mark = cls.get_mark_at_previous_review(obj)
        return cls.cast_mark_to_num(obj, mark)

    @classmethod
    def get_mark_before_the_previous_review(cls, obj):
        return cls._get_mark_from_history(obj, -2)

    @classmethod
    def get_mark_before_the_previous_review_as_number(cls, obj):
        mark = cls.get_mark_before_the_previous_review(obj)
        return cls.cast_mark_to_num(obj, mark)

    @classmethod
    def get_mark_at_review(cls, obj):
        scale = obj.review_marks_scale.get(obj.mark)
        if isinstance(scale, dict):
            mark = scale.get('text_value')
            if mark is not None:
                return mark

        return cls._mark_to_str(obj.mark)

    @classmethod
    def get_mark_at_review_as_number(cls, obj):
        return cls.cast_mark_to_num(obj, obj.mark)

    @classmethod
    def get_salary_change_absolute(self, obj):
        val = obj.salary_change_absolute
        return val if isinstance(val, core_const.SPECIAL_VALUE) else Decimal(val)

    @classmethod
    def get_bonus_option(cls, obj):
        return bool(obj.options_rsu)

    @classmethod
    def get_extra_payment(cls, obj):
        if isinstance(obj.goldstar, core_const.SPECIAL_VALUE):
            return obj.goldstar
        return legacy.extra_payment_from_goldstar(obj.goldstar)

    @classmethod
    def get_extra_option(cls, obj):
        if isinstance(obj.goldstar, core_const.SPECIAL_VALUE):
            return obj.goldstar
        return legacy.extra_option_from_goldstar(obj.goldstar)

    @classmethod
    def get_extra_payment_at_previous_review(cls, obj):
        if not obj.short_history:
            return
        previous = obj.short_history[-1]
        return legacy.extra_payment_from_goldstar(previous['goldstar'])

    @classmethod
    def get_extra_option_at_previous_review(cls, obj):
        if not obj.short_history:
            return
        previous = obj.short_history[-1]
        return legacy.extra_option_from_goldstar(previous['goldstar'])

    @classmethod
    def get_full_department_name(cls, obj):
        return ' / '.join(obj.person_department_chain_names)

    @classmethod
    def get_review_link(cls, obj):
        protocol = settings.REVIEW_FRONTEND['protocol']
        host = settings.REVIEW_FRONTEND['host']
        return '%s://%s/reviews/%s/persons/%s' % (
            protocol,
            host,
            obj.review_id,
            obj.person_login,
        )

    @classmethod
    def get_feedback_link(cls, obj):
        protocol = settings.FEEDBACK_FRONTEND['protocol']
        host = settings.FEEDBACK_FRONTEND['host']
        person_id = obj.person_id
        return '%s://%s/people?user=%s' % (protocol, host, person_id)

    @classmethod
    def get_goals_link(cls, obj):
        protocol = settings.GOALS_FRONTEND['protocol']
        host = settings.GOALS_FRONTEND['host']
        person_id = obj.person_id
        return '%s://%s/filter?user=%s' % (protocol, host, person_id)

    @classmethod
    def get_main_product(cls, obj):
        if not obj.main_product:
            return None

        return obj.main_product['name']

    @classmethod
    def get_umbrella(cls, obj):
        if not obj.umbrella:
            return None

        if obj.umbrella['main_product']:
            return '%s / %s' % (obj.umbrella['main_product']['name'], obj.umbrella['name'])
        else:
            return obj.umbrella['name']

    @classmethod
    def get_updated_at(cls, obj):
        if obj:
            return obj.updated_at.strftime('%Y-%m-%d %H:%M')
