import json
import logging
from collections import defaultdict
from typing import Any, Dict, List, Optional

import attr
from celery import states
from django_celery_results.models import TaskResult
from django.db.models import Q

from review.core import (
    const,
    models,
    tasks as core_tasks,
)
from review.core.logic import assemble
from review.core.task_handlers import TaskInfo, get_pedning_event_type_by_task, PENDING_EVENT_TYPES
from review.staff import models as staff_models


log = logging.getLogger(__name__)


RUNNNED_STATES = frozenset({
    states.STARTED,
    states.RETRY,
    states.IGNORED,
    states.RECEIVED,
    states.PENDING,
})


@attr.s
class PersonReviewTableTasks:
    grant_permissions = attr.ib(factory=list, type=List['str'])
    status_change = attr.ib(factory=list, type=List['str'])


def _json_loads_repr_dict(d: Optional[str]) -> Dict[str, Any]:
    if d is None:
        return {}
    # HACK: only works if keys are not strings with replacing values
    return json.loads(
        d
        .replace("'", '"')
        .replace('True', 'true')
        .replace('False', 'false')
        .replace('None', 'null')
    )


def get_running_tasks(task_types: List[str]) -> List[TaskInfo]:
    qs = (
        TaskResult.objects
        .filter(
            task_name__in=task_types,
            status__in=RUNNNED_STATES,
        )
        .values(
            'task_id',
            'task_kwargs',
            'task_name',
        )
    )
    return [
        TaskInfo(
            it['task_id'],
            it['task_name'],
            _json_loads_repr_dict(it['task_kwargs']),
        )
        for it in qs
    ]


def get_review_participants_pending_tasks(
    subject: staff_models.Person,
    review_id: int,
) -> List[str]:
    tasks = get_running_tasks([core_tasks.bulk_same_action_set_task.name])
    res = _get_reviewers_changing_tasks(
        subject,
        review_id,
        tasks,
    )
    log.info(
        f'Pending tasks for {subject.login} '
        f'at review/participants {review_id} are: {res}'
    )
    return res


def get_active_reviews_pending_tasks(
    subject: staff_models.Person,
) -> PersonReviewTableTasks:
    available_ids = assemble.get_person_review_ids(
        subject,
        {const.FILTERS.REVIEW_ACTIVITY: True},
        const.ROLE.PERSON_REVIEW_LIST_RELATED,
    )
    tasks = get_running_tasks([
        core_tasks.bulk_same_action_set_task.name,
        core_tasks.denormalize_person_review_roles_task.name,
    ])
    res = _get_tasks_for_person_reviews(tasks, available_ids)
    log.info(
        f'Pending tasks for {subject.login} '
        f'at review/active are: {res}'
    )
    return res


def get_review_pending_tasks(
    subject: staff_models.Person,
    review_id: int,
) -> PersonReviewTableTasks:
    available_ids = assemble.get_person_review_ids(
        subject,
        {const.FILTERS.REVIEWS: [review_id]},
        const.ROLE.PERSON_REVIEW_LIST_RELATED,
    )
    tasks = get_running_tasks([
        core_tasks.bulk_same_action_set_task.name,
        core_tasks.denormalize_person_review_roles_task.name,
    ])
    res = _get_tasks_for_person_reviews(tasks, available_ids)
    log.info(
        f'Pending tasks for {subject.login} '
        f'at review {review_id} are: {res}'
    )
    return res


def get_calibration_pending_tasks(
    subject: staff_models.Person,
    calibration_id: int,
) -> PersonReviewTableTasks:
    available_ids = assemble.get_person_review_ids(
        subject,
        {const.FILTERS.CALIBRATIONS: [calibration_id]},
        const.ROLE.CALIBRATION.ALL,
    )
    tasks = get_running_tasks([
        core_tasks.bulk_same_action_set_task.name,
        core_tasks.denormalize_person_review_roles_task.name,
    ])
    res = _get_tasks_for_person_reviews(tasks, available_ids)
    log.info(
        f'Pending tasks for {subject.login} '
        f'at calibration {calibration_id} are: {res}'
    )
    return res


def _get_reviewers_changing_tasks(
    subject: staff_models.Person,
    review_id: int,
    tasks: List[TaskInfo],
) -> List[str]:
    reviewers_chaning_pr_ids = set()
    id_to_task = defaultdict(list)
    for task in tasks:
        if get_pedning_event_type_by_task(task) != PENDING_EVENT_TYPES.REVIEWERS_CHANGE:
            continue
        reviewers_chaning_pr_ids |= set(task.kwargs['ids'])
        for pr_id in task.kwargs['ids']:
            id_to_task[pr_id].append(task.id)

    available_ids = assemble.get_person_review_ids(
        subject,
        {
            const.FILTERS.IDS: reviewers_chaning_pr_ids,
            const.FILTERS.REVIEWS: [review_id],
        },
        const.ROLE.REVIEW.ALL,
    )

    running_task_ids = set()
    for id_ in available_ids:
        running_task_ids |= set(id_to_task[id_])

    return list(running_task_ids)


def _get_tasks_for_person_reviews(
    tasks: List[TaskInfo],
    person_review_ids: List[int],
) -> PersonReviewTableTasks:
    pr_id_to_roles_tasks = defaultdict(list)
    pr_id_to_status_change_tasks = defaultdict(list)
    review_id_to_tasks = defaultdict(list)
    calibration_id_to_tasks = defaultdict(list)
    for task in tasks:
        pr_filter = task.kwargs.get('person_review_ids') or []
        cal_id = task.kwargs.get('calibration_id')
        review_id = task.kwargs.get('review_id')
        bulk_params = task.kwargs.get('params')
        for pr_id in pr_filter:
            pr_id_to_roles_tasks[pr_id].append(task.id)
        if cal_id:
            calibration_id_to_tasks[cal_id].append(task.id)
        if review_id:
            review_id_to_tasks[review_id].append(task.id)
        if bulk_params:
            if get_pedning_event_type_by_task(task) == PENDING_EVENT_TYPES.STATUS_CHANGE:
                for id_ in task.kwargs['ids']:
                    pr_id_to_status_change_tasks[id_].append(task.id)

    if not any((
        pr_id_to_roles_tasks,
        review_id_to_tasks,
        calibration_id_to_tasks,
        pr_id_to_status_change_tasks,
    )):
        return PersonReviewTableTasks()
    task_person_review_filter = Q()
    if pr_id_to_roles_tasks or pr_id_to_status_change_tasks:
        pr_ids = list(pr_id_to_status_change_tasks) + list(pr_id_to_roles_tasks)
        task_person_review_filter |= Q(id__in=pr_ids)
    if review_id_to_tasks:
        task_person_review_filter |= Q(review_id__in=review_id_to_tasks)
    if calibration_id_to_tasks:
        task_person_review_filter |= Q(
            calibration_person_review__calibration_id__in=calibration_id_to_tasks
        )

    person_review_ids_to_return = (
        models.PersonReview.objects
        .filter(task_person_review_filter)
        .filter(id__in=person_review_ids)
        .values_list(
            'id',
            'review_id',
            'calibration_person_review__calibration_id',
        )
    )

    grant_permissions_tasks = set()
    status_change_tasks = set()
    for pr_id, review_id, calibration_id in person_review_ids_to_return:
        grant_permissions_tasks |= set(pr_id_to_roles_tasks[pr_id])
        grant_permissions_tasks |= set(review_id_to_tasks[review_id])
        grant_permissions_tasks |= set(calibration_id_to_tasks[calibration_id])
        status_change_tasks |= set(pr_id_to_status_change_tasks[pr_id])
    return PersonReviewTableTasks(
        list(grant_permissions_tasks),
        list(status_change_tasks),
    )
