import attr
from celery import states
from typing import Any, Dict, List, Optional

from review.core import models, tasks as core_tasks
from review.staff.models import Person
from review.xiva.pending_tasks import post_task_status_to_xiva


class PENDING_EVENT_TYPES:
    REVIEWERS_CHANGE = 'reviewers_change'
    GRANT_PERMISSIONS = 'grant_permissions'
    STATUS_CHANGE = 'status_change'
    BULK_CHANGE = 'bulk_change'


PERSON_REVIEW_ACTIONS = frozenset(('approve', 'unapprove', 'allow_announce', 'announce'))


@attr.s
class TaskInfo:
    id = attr.ib(type=str)
    name = attr.ib(type=str)
    kwargs = attr.ib(type=Dict[str, Any], factory=dict)


def get_pedning_event_type_by_task(task_info: TaskInfo) -> Optional[str]:
    event = None

    if task_info.name == core_tasks.bulk_same_action_set_task.name:
        bulk_params = task_info.kwargs.get('params', {})

        if 'reviewers' in bulk_params:
            event = PENDING_EVENT_TYPES.REVIEWERS_CHANGE
        elif bulk_params.keys() & PERSON_REVIEW_ACTIONS:
            event = PENDING_EVENT_TYPES.STATUS_CHANGE
        else:
            event = PENDING_EVENT_TYPES.BULK_CHANGE

    elif task_info.name == core_tasks.denormalize_person_review_roles_task.name:
        event = PENDING_EVENT_TYPES.GRANT_PERMISSIONS

    return event


def _get_review_ids_by_person_reviews(person_review_ids: List[int]) -> List[int]:
    review_ids = (
        models.PersonReview.objects
        .filter(id__in=person_review_ids)
        .values_list('review_id', flat=True)
        .distinct('review_id')
    )
    return list(review_ids)


def _get_calibration_ids_by_person_review(person_review_ids: List[int]) -> List[int]:
    calibration_ids = (
        models.CalibrationPersonReview.objects
        .filter(person_review_id__in=person_review_ids)
        .values_list('calibration_id', flat=True)
        .distinct('calibration_id')
    )
    return list(calibration_ids)


def get_xiva_tags(review_ids=None, calibration_ids=None):
    tags = [f'review_{review_id}' for review_id in review_ids or []]
    tags.extend(f'calibration_{calibration_id}' for calibration_id in calibration_ids or [])
    return tags


def bulk_same_action_set_task_status_handler(task_info: TaskInfo, status: str) -> None:
    event = get_pedning_event_type_by_task(task_info)
    if not event or not task_info.kwargs.get('ids'):
        return

    review_ids = _get_review_ids_by_person_reviews(task_info.kwargs.get('ids'))
    message = {
        'task_id': task_info.id,
        'status': status,
        'actions': list(task_info.kwargs.get('params', {}).keys()),
    }
    subject_uid = Person.objects.get(id=task_info.kwargs.get('subject_id')).uid
    post_task_status_to_xiva(subject_uid, event, message, tags=get_xiva_tags(review_ids))


def bulk_same_action_set_task_success_handler(task, retval, task_id, args, kwargs):
    task_info = TaskInfo(id=task_id, name=task.name, kwargs=kwargs)
    bulk_same_action_set_task_status_handler(task_info, states.SUCCESS.lower())


def bulk_same_action_set_task_failure_handler(task, exc, task_id, args, kwargs, einfo):
    task_info = TaskInfo(id=task_id, name=task.name, kwargs=kwargs)
    bulk_same_action_set_task_status_handler(task_info, states.FAILURE.lower())
