# coding: utf-8
import logging
from datetime import timedelta
from typing import List

import psycopg2
import yenv
from celery import shared_task
from django.db import IntegrityError
from django.db.utils import OperationalError
from django.conf import settings
from django.core.exceptions import ValidationError
from django.core.mail import EmailMessage
from django.db import transaction
from django.utils.timezone import now
from django_celery_monitoring.utils import fail_aborted_celery_results
from django_celery_results.models import TaskResult
from post_office import mail
from ylog.context import log_context

from ids.exceptions import BackendError

from review.core import const
from review.core.logic import assemble, review_actions, roles, bulk
from review.core.logic.export import get_properties, get_serialized
from review.core.models import PersonReview, Kpi, Review
from review.lib import datetimes
from review.lib import lock
from review.oebs.sync.fetch import OebsConnector
from review.staff.models import Person
from review.core import task_handlers


log = logging.getLogger(__name__)


@lock.get_lock_or_do_nothing_task
def update_role_inheritance():
    # if this returns not {'deleted': 0, 'created': 0} there are a bugs
    # in roles denormalization
    return roles.denormalize_person_review_roles()


@lock.get_lock_or_do_nothing_task
def finish_reviews_task():
    return review_actions.finish_reviews()


@shared_task
def publish_review_results(review_id):
    return review_actions.publish_review_results(review_id)


@shared_task
def add_global_admins():
    return review_actions.add_global_admins()


@lock.get_lock_or_do_nothing_task
def fixing_bonus_type():
    PersonReview.objects.filter(
        review_id__in=(955, 953, 947, 946, 945, 944,
                       943, 942, 941, 940, 939, 938,
                       937, 936, 935, 934, 933, 932)
    ).update(bonus_type=const.SALARY_DEPENDENCY_TYPE.PERCENTAGE)


@lock.get_lock_or_do_nothing_task
def follow_workflow(user_login, review_id, new_workflow):
    user = Person.objects.get(login=user_login)
    review = assemble.get_review(
        subject=user,
        filters={'ids': [review_id]},
    )
    review_actions.follow_workflow(
        review=review,
        new_workflow=new_workflow,
    )


@shared_task(bind=True, max_retries=5)
def denormalize_person_review_roles_task(
    self,
    person_review_ids: List[int]=None,
    review_id: int=None,
    person_ids: List[int]=None,
    calibration_id: int=None,
    skip_calibration: bool=False,
):
    try:
        return roles.denormalize_person_review_roles(
            person_review_ids=person_review_ids,
            review_id=review_id,
            person_ids=person_ids,
            calibration_id=calibration_id,
            skip_calibration=skip_calibration,
        )
    except (
        # duplicate errors
        psycopg2.errors.UniqueViolation,
        IntegrityError,
        # lock timeout
        OperationalError,
    ) as exc:
        if self.request.retries == self.max_retries:
            raise
        else:
            raise self.retry(exc=exc, eta=now() + timedelta(seconds=20))


@lock.get_lock_or_do_nothing_task
def load_product_scheme(review_id, date_from, date_to):
    raise Exception('Remove old sync')


@shared_task(bind=True, max_retries=10)
def export_person_review_template_task(self, user_login, data, fields):
    user = Person.objects.get(login=user_login)

    review_id = data.pop('id')
    data_template_type = data.pop('template_type')
    template_type = const.EXCEL_TEMPLATE_TYPE.REVERSE_VERBOSE[data_template_type]

    properties = get_properties(review_id=review_id, template_type=template_type, user=user)

    if yenv.type not in ('production', 'prestable'):
        recipients = [settings.DEBUG_EMAIL]
    else:
        recipients = [user.work_email]

    try:
        attachment_bytes = get_serialized(
            data=data,
            user=user,
            review_id=review_id,
            fields=fields,
            properties=properties,
        )

        email = EmailMessage(
            subject='Export for review {}'.format(review_id),
            to=recipients,
            from_email=settings.DEFAULT_FROM_EMAIL,
        )

        email.attach('exported.xls', attachment_bytes, 'application/vnd.ms-excel')
        email.send()

    except Exception as e:
        log.error(e.message)

        if self.request.retries == self.max_retries:
            mail.send(
                recipients=recipients,
                sender=settings.DEFAULT_FROM_EMAIL,
                subject='Export for review {}'.format(review_id),
                message="Error while process review export. Please, contact Team OK for more information"
            )
        else:
            data['id'] = review_id
            data['template_type'] = data_template_type
            raise self.retry(exc=e, eta=now() + timedelta(seconds=60))


@shared_task(bind=True, max_retries=10)
@transaction.atomic
def load_kpi(self, review_id, date_from, date_to):
    robot = Person.objects.get(login='robot-review')
    person_reviews = assemble.get_person_reviews(
        subject=robot,
        filters_chosen={
            const.FILTERS.REVIEWS: [review_id],
        },
        fields_requested=(
            const.FIELDS.ID,
            const.FIELDS.PERSON_LOGIN,
        )
    )
    login_to_person_reviews = {
        person_review.person_login: person_review.id
        for person_review in person_reviews
    }
    try:
        connector = OebsConnector()

        oebs_reward_data = connector.post(resource='kpi', json={
            'login': list(login_to_person_reviews),
            'startDate': date_from,
            'finishDate': date_to,
            'periodType': ['quarterly'],
        })
    except BackendError as exc:
        response_content = exc.response is not None and exc.response.content
        status_code = exc.response is not None and exc.response.status_code
        log.exception(
            'Oebs bad response %s `%s`. data_type=kpi, logins=%s',
            status_code,
            response_content,
            list(login_to_person_reviews),
        )
        raise self.retry(eta=now() + timedelta(seconds=60))

    Kpi.objects.filter(person_review_id__in=login_to_person_reviews.values()).delete()
    kpi_list = []
    for period in oebs_reward_data.get('Rewards') or []:
        for person_data in period.get('personAnalytics') or []:
            for goal in person_data.get('goals') or []:
                year, quarter = map(int, period['periodName'].split('_Q'))
                kpi_list.append(
                    Kpi(
                        person_review_id=login_to_person_reviews[person_data['login']],
                        name=goal['goalsName'],
                        year=year,
                        quarter=quarter,
                        percent=goal['prcOfCompletion'],
                        weight=goal['weight'],
                    )
                )
    Kpi.objects.bulk_create(kpi_list)
    Review.objects.filter(id=review_id).update(kpi_loaded=datetimes.now())


@shared_task
def clean_old_celery_results_task():
    dt = now() - timedelta(days=settings.CELERY_MONITORING_EXPIRE_DAYS)
    dt = dt.replace(hour=0, minute=0, second=0, microsecond=0)
    qs = TaskResult.objects.filter(
        date_done__lt=dt,
    )
    qs.delete()


@shared_task
def fail_aborted_celery_results_task():
    fail_aborted_celery_results()


@shared_task(
    bind=True,
    on_failure=task_handlers.bulk_same_action_set_task_failure_handler,
    on_success=task_handlers.bulk_same_action_set_task_success_handler,
)
def bulk_same_action_set_task(
    self,
    subject_id,
    ids,
    params,
    subject_type=const.PERSON_REVIEW_CHANGE_TYPE.PERSON,
):
    context = {
        'task_id': self.request.id,
        'task_name': self.name,
    }
    from review.frontend.views.actions import ActionParams
    action_form = ActionParams(data=params)
    if not action_form.is_valid():
        raise ValidationError(action_form.errors)
    params = action_form.cleaned_data

    with log_context(**context):
        subject = Person.objects.get(id=subject_id)
        bulk_result = bulk.bulk_same_action_set(subject, ids, params, subject_type)
        result = [(pr.id, res) for pr, res in bulk_result.items()]
        return result


@shared_task
def freeze_gradient_data_task(review_id, freezing_datetime, person_review_ids=None):
    from review.gradient.freezing import freeze_gradient_data
    freeze_gradient_data(review_id, freezing_datetime, person_review_ids)
