# coding: utf-8
from collections import OrderedDict

from django.conf import settings

from django.core import urlresolvers
from review.shortcuts import const
from review.core.logic.assemble import permissions_person_review
from review.core.logic import calibration_rights


GLOBAL_PERMS = const.GLOBAL_ACTIONS
REVIEW_PERMS = const.REVIEW_PERMS
PERSON_REVIEW_PERMS = const.PERSON_REVIEW_PERMS
CALIBRATION_PERMS = calibration_rights.PERMS

VERBOSE_GLOBAL_PERMS = OrderedDict([
    (GLOBAL_PERMS.CREATE_REVIEW, 'создавать ревью'),
    (GLOBAL_PERMS.CREATE_CALIBRATION, 'создавать калибровку'),
    (GLOBAL_PERMS.ADD_PERSON_REVIEWS_TO_CALIBRATION, 'добавлять людей в калибровку'),
])


VERBOSE_REVIEW_PERMS = OrderedDict([
    (REVIEW_PERMS.LIST_SUPERREVIEWERS, 'видеть список суперревьюеров'),
    (REVIEW_PERMS.LIST_ADMINS, 'видеть список администраторов'),
    (REVIEW_PERMS.LIST_ACCOMPANYING_HRS, 'видеть список сопровождающих hr'),
    (REVIEW_PERMS.LIST_GOODIES, 'видеть загруженные плюшки ревью'),
    (REVIEW_PERMS.ADD_ANY_PERSONS, 'добавлять любых участников'),
    (REVIEW_PERMS.ADD_PERSONS, 'добавлять участников, доступных по hr правам'),
    (REVIEW_PERMS.DELETE_ANY_PERSONS, 'удалять любых участников'),
    (REVIEW_PERMS.DELETE_ANY_PERSONS, 'удалять участников, доступных по hr правам'),
    (REVIEW_PERMS.EDIT, 'редактировать параметры'),
    (REVIEW_PERMS.STATUS_PUBLISH, 'статус: запустить'),
    (REVIEW_PERMS.STATUS_IN_DRAFT, 'статус: в черновик'),
    (REVIEW_PERMS.STATUS_ARCHIVE, 'статус: архивировать'),
])


VERBOSE_PERSON_REVIEW_PERMS = OrderedDict([
    (PERSON_REVIEW_PERMS.MARK_READ, 'чтение: оценка'),
    (PERSON_REVIEW_PERMS.GOLDSTAR_READ, 'чтение: голдстар'),
    (PERSON_REVIEW_PERMS.STATUS_READ, 'чтение: статус'),
    (PERSON_REVIEW_PERMS.MARK_LEVEL_HISTORY_READ, 'чтение: короткая история'),
    (PERSON_REVIEW_PERMS.APPROVE_LEVEL_READ, 'чтение: уровень апрува'),
    (PERSON_REVIEW_PERMS.LEVEL_CHANGE_READ, 'чтение: изменение уровня'),
    (PERSON_REVIEW_PERMS.SALARY_CHANGE_READ, 'чтение: изменение зарплаты'),
    (PERSON_REVIEW_PERMS.BONUS_READ, 'чтение: зарплата'),
    (PERSON_REVIEW_PERMS.OPTIONS_RSU_READ, 'чтение: опционы'),
    (PERSON_REVIEW_PERMS.FLAGGED_READ, 'чтение: флажок'),
    (PERSON_REVIEW_PERMS.COMMENTS_READ, 'чтение: комментарии'),
    (PERSON_REVIEW_PERMS.CHANGES_READ, 'чтение: изменения'),
    (PERSON_REVIEW_PERMS.FTE_READ, 'чтение: ставка'),
    (PERSON_REVIEW_PERMS.SALARY_READ, 'чтение: зарплата перед ревью'),
    (PERSON_REVIEW_PERMS.LEVEL_READ, 'чтение: уровень перед ревью'),
    (PERSON_REVIEW_PERMS.REVIEWERS_READ, 'чтение: цепочка ревьюеров'),
    (PERSON_REVIEW_PERMS.ACTION_AT_READ, 'чтение: на ком действие'),
    (PERSON_REVIEW_PERMS.MARK_WRITE, 'запись: оценка'),
    (PERSON_REVIEW_PERMS.GOLDSTAR_WRITE, 'запись: голдстар ручной'),
    (PERSON_REVIEW_PERMS.GOLDSTAR_WRITE_CHOSEN, 'запись: голдстар для верховного ревьюера'),
    (PERSON_REVIEW_PERMS.LEVEL_CHANGE_WRITE, 'запись: изменение уровня'),
    (PERSON_REVIEW_PERMS.SALARY_CHANGE_WRITE, 'запись: изменение зарплаты'),
    (PERSON_REVIEW_PERMS.BONUS_WRITE, 'запись: премия'),
    (PERSON_REVIEW_PERMS.OPTIONS_RSU_WRITE, 'запись: опционы'),
    (PERSON_REVIEW_PERMS.SALARY_CHANGE_AUTO_WRITE, 'запись: ручное изменение авто зарплаты'),
    (PERSON_REVIEW_PERMS.BONUS_AUTO_WRITE, 'запись: ручное изменение авто премии'),
    (PERSON_REVIEW_PERMS.OPTIONS_RSU_AUTO_WRITE, 'запись: ручное изменение авто опционов'),
    (PERSON_REVIEW_PERMS.FLAGGED_WRITE, 'запись: флажок'),
    (PERSON_REVIEW_PERMS.COMMENTS_WRITE, 'запись: комментарий'),
    (PERSON_REVIEW_PERMS.REVIEWERS_WRITE, 'запись: цепочка ревьюеров'),
    (PERSON_REVIEW_PERMS.APPROVE, 'апрувить'),
    (PERSON_REVIEW_PERMS.UNAPPROVE, 'отменить апрув'),
    (PERSON_REVIEW_PERMS.ALLOW_ANNOUNCE, 'разрешить объявлять'),
    (PERSON_REVIEW_PERMS.ANNOUNCE, 'объявить'),
])


VERBOSE_CALIBRATION_PERMS = OrderedDict([
    (CALIBRATION_PERMS.EDIT_CALIBRATION_PERSON_REVIEWS, 'редактировать состав калибруемых'),
    (CALIBRATION_PERMS.EDIT_CALIBRATORS, 'редактировать состав калибрующих'),
    (CALIBRATION_PERMS.EDIT_PARAMETERS, 'редактировать параметры калибровки'),
    (CALIBRATION_PERMS.LIST_CALIBRATORS, 'видеть список калибрующих'),
    (CALIBRATION_PERMS.LIST_PERSON_REVIEWS, 'видеть список калибруемых'),
    (CALIBRATION_PERMS.READ_FEEDBACK, 'видеть отзывы'),
    (CALIBRATION_PERMS.STATUS, 'видеть статус'),
    (CALIBRATION_PERMS.DELETE, 'удалять калибровку'),
])


VERBOSE_PERSON_PERMS = OrderedDict([
    ('finance', 'все оебс-данные'),
])


VERBOSE_DYNAMIC_PERM = {
    'person_review_perm_for_superreviewer_write':
        'Если ревью активно И PersonReview '
        'не в статусах «Анонсировано» или «Ожидает анонса»',
    'check_person_review_transition':
        'Если существует '
        '((https://wiki.yandex-team.ru/cia/revju-2.0/flow/ воркфлоу))-переход',
    'person_review_as_self_read': 'Если PersonReview в статусе «Анонсировано»',
    'person_review_perm_for_reviewer_write':
        'Если Ревью активно И PersonReview '
        'не в статусах «Анонсировано» или «Ожидает анонса» '
        'И положение ревьюера в цепочке выше '
        'или равно текущему указателю в цепочке',
    'is_active_not_announced':
        'Если Ревью активно И PersonReview не «Анонсировано»',
    'reviewer_can_approve':
        'Если ревью активно И существует воркфлоу-переход '
        'И положение ревьюера в цепочке выше '
        'или равно текущему указателю в цепочке',
    'top_reviewer_can_unapprove':
        'Если ревью активно И существует воркфлоу-переход '
        'И PersonReview в статусе «Согласовано»',
    'reviewer_can_unapprove':
        'Если ревью активно И PersonReview '
        'И существует воркфлоу-переход '
        'И текущий указатель на следующем после положение ревьюера в цепочке '
        'И PersonReview в статусе «Согласование»',
    'is_calibration_admin': 'Если является админом хотя бы одной калибровки',
}


SCOPES = (
    const.ROLE.GLOBAL,
    const.ROLE.REVIEW,
    const.ROLE.PERSON_REVIEW,
    const.ROLE.PERSON,
    const.ROLE.CALIBRATION,
    const.ROLE.DEPARTMENT,
)
ALL_ROLES_VERBOSE = OrderedDict([
    (const.ROLE.GLOBAL.REVIEW_CREATOR, 'Создатель ревью'),
    (const.ROLE.GLOBAL.CALIBRATION_CREATOR, 'Создатель калибровок'),
    (const.ROLE.GLOBAL.SUPPORT, 'Саппорт'),
    (const.ROLE.GLOBAL.EXPORTER, 'Выгружающий'),
    (const.ROLE.GLOBAL.ROBOT, 'Робот'),
    (const.ROLE.GLOBAL.ADMIN, 'Глобальный админ'),
    (const.ROLE.GLOBAL.BI_VIEWER, 'Смотрящий в BI'),
    (const.ROLE.GLOBAL.LOAN_VIEWER, 'Смотрящий в данные по ссудам'),
    (const.ROLE.REVIEW.ADMIN, 'Администратор ревью'),
    (const.ROLE.REVIEW.SUPERREVIEWER, 'Суперревьюер'),
    (const.ROLE.REVIEW.ACCOMPANYING_HR, 'Сопровождающий hr'),
    (const.ROLE.CALIBRATION.ADMIN, 'Администратор калибровки'),
    (const.ROLE.CALIBRATION.CALIBRATOR, 'Калибрующий'),
    (const.ROLE.PERSON_REVIEW.REVIEWER, 'Ревьюер'),
    (const.ROLE.PERSON_REVIEW.READER, 'Читатель'),
    (const.ROLE.PERSON_REVIEW.SUPERREADER, 'Суперчитатель'),
    (const.ROLE.PERSON_REVIEW.TOP_REVIEWER, 'Финальный утверждающий'),
    (const.ROLE.PERSON_REVIEW.READER_INHERITED, 'Читатель для прошлых ревью'),
    (const.ROLE.PERSON_REVIEW.SUPERREADER_INHERITED, 'Суперчитатель для прошлых ревью'),
    (const.ROLE.PERSON_REVIEW.REVIEWER_INHERITED, 'Ревьюер для прошлых ревью'),
    (const.ROLE.PERSON_REVIEW.TOP_REVIEWER_INHERITED, 'Финальный утверждающий для прошлых ревью'),
    (const.ROLE.PERSON_REVIEW.SUPERREVIEWER_INHERITED, 'Суперревьюер для прошлых ревью'),
    (const.ROLE.PERSON_REVIEW.CALIBRATION_ADMIN_INHERITED, 'Администратор калибровки для прошлых ревью'),
    (const.ROLE.PERSON_REVIEW.CALIBRATOR_INHERITED, 'Калибрующий для прошлых ревью'),
    (const.ROLE.PERSON_REVIEW.CALIBRATOR_ARCHIVED, 'Калибрующий архивный'),
    (const.ROLE.PERSON.SELF, 'Сам сотрудник'),
    (const.ROLE.DEPARTMENT.HEAD, 'Руководитель'),
    (const.ROLE.DEPARTMENT.HR_PARTNER, 'HR-партнер'),
    (const.ROLE.DEPARTMENT.HR_ANALYST, 'HR-аналитик'),
    (const.ROLE.DEPARTMENT.FINANCE_VIEWER, 'HR смотритель финансов'),
])

ALL_ROLES_ORDER = [_role_id for _role_id in ALL_ROLES_VERBOSE]


def roles_sorter(role_id):
    return ALL_ROLES_ORDER.index(role_id)


READER_ROLES = {
    const.ROLE.PERSON_REVIEW.READER,
    const.ROLE.PERSON_REVIEW.READER_INHERITED,
    const.ROLE.PERSON_REVIEW.SUPERREADER,
    const.ROLE.PERSON_REVIEW.SUPERREADER_INHERITED,
}
DENORMALIZED = set(const.ROLE.DENORMALIZATION_REVERSED)
FROM_PERSON_REVIEW_TO_REVIEW = {
    const.ROLE.PERSON_REVIEW.SUPERREVIEWER_INHERITED,
}
FROM_PERSON_REVIEW_TO_CALIBRATION = {
    const.ROLE.PERSON_REVIEW.CALIBRATOR_ARCHIVED,
    const.ROLE.PERSON_REVIEW.CALIBRATOR_INHERITED,
    const.ROLE.PERSON_REVIEW.CALIBRATION_ADMIN_INHERITED,
}
TABLE_BLOCKS = OrderedDict([
    ('Глобальные', const.ROLE.GLOBAL.ALL),
    (
        'Привязанные к ревью',
        const.ROLE.REVIEW.ALL | FROM_PERSON_REVIEW_TO_REVIEW
    ),
    (
        'Привязанные к калибровке',
        const.ROLE.CALIBRATION.ALL | FROM_PERSON_REVIEW_TO_CALIBRATION
    ),
    (
        'Привязанные к ревьюимому I',
        const.ROLE.PERSON_REVIEW.ALL -
        READER_ROLES -
        DENORMALIZED -
        FROM_PERSON_REVIEW_TO_REVIEW -
        FROM_PERSON_REVIEW_TO_CALIBRATION
    ),
    ('Привязанные к ревьюимому II', READER_ROLES),
    ('Привязанные к подразделению', const.ROLE.DEPARTMENT.ALL),
    ('Привязанные к сотруднику', const.ROLE.PERSON.ALL),
])

PERMS_GROUPS = OrderedDict([
    ('Пермишны глобальные', VERBOSE_GLOBAL_PERMS),
    ('Пермишны для ревью', VERBOSE_REVIEW_PERMS),
    ('Пермишны для калибровки', VERBOSE_CALIBRATION_PERMS),
    ('Пермишны для ревьюируемого', VERBOSE_PERSON_REVIEW_PERMS),
    ('Пермишны для сотрудника', VERBOSE_PERSON_PERMS),
])
PERM_GROUP_TO_ROLE_BLOCKS = {
    'Глобальные': {
        'Пермишны глобальные',
        'Пермишны для ревью',
        'Пермишны для ревьюируемого',
    },
    'Привязанные к ревью': {
        'Пермишны для ревью',
        'Пермишны для ревьюируемого',
    },
    'Привязанные к ревьюимому I': {
        'Пермишны для ревьюируемого',
    },
    'Привязанные к ревьюимому II': {
        'Пермишны для ревьюируемого',
    },
    'Привязанные к калибровке': {
        'Пермишны для калибровки',
        'Пермишны для ревьюируемого',
    },
    'Привязанные к подразделению': {
        'Пермишны для ревьюируемого',
        'Пермишны для сотрудника',
    },
    'Привязанные к сотруднику': {
        'Пермишны для ревьюируемого',
        'Пермишны для сотрудника',
    },
}

PERM_GROUP_BLOCK_COMMENT = {
    ('Глобальные', 'Пермишны для ревью'): '\n'.join([
        '(для любого ревью)',
    ]),
    ('Привязанные к ревью', 'Пермишны для ревью'): '\n'.join([
        '(по отношению к ревью, на которое выдана роль)',
    ]),
    ('Привязанные к ревью', 'Пермишны для ревьюируемого'): '\n'.join([
        '(для всех ревьюируемых в ревью, на которое выдана роль, кроме своего)',
    ]),
    ('Привязанные к калибровке', 'Пермишны для калибровки'): '\n'.join([
        '(по отношению к калибровке, на которую выдана роль)',
    ]),
    ('Привязанные к калибровке', 'Пермишны для ревьюируемого'): '\n'.join([
        '(для всех ревьюируемых в калибровке, на которую выдана роль, кроме своего)',
    ]),
    ('Привязанные к ревьюимому I', 'Пермишны для ревьюируемого'): '\n'.join([
        '(по отношению к ревьируемого, на которого выдана роль через цепочку апруверов, кроме своего)',
    ]),
    ('Привязанные к ревьюимому II', 'Пермишны для ревьюируемого'): '\n'.join([
        '(по отношению к ревьируемого, на которого выдана роль через админку, кроме своего)',
    ]),
    ('Привязанные к подразделению', 'Пермишны для ревьюируемого'): '\n'.join([
        '(по отношению к PersonReview, где person находится в подразделении, '
        'к которому на данный момент по стаффу привязана роль)',
    ]),
    ('Привязанные к сотруднику', 'Пермишны для ревьюируемого'): '\n'.join([
        '(по отношению к PersonReview, где person совпадает с person, '
        'к которому привязана роль)',
    ]),
}

PERMISSIONS_API_TEXT = '\n'.join([
    'Можно посмотреть к каким ревью/person_review у пользователя есть доступ ',
    '%s/dev/perms/<username>/ ' % settings.BACKEND_URL,
])


YES = '!!(зел)✔!!'
MAYBE_TPL = '!!(жел)%s!!'
STAR = '★'
NO = '!!✖!!'


def build_docs_roles_wiki_markup():
    pieces = [
        build_header('Таблицы пермишнов (автоматические)', level=2),
        build_page_start(),
        build_header('Пермишны по ролям'),
    ]

    for block_name, block_roles in TABLE_BLOCKS.items():
        pieces.append(
            build_table_for_roles(role_block_name=block_name, roles=block_roles)
        )

    pieces.append(build_roles_lists())

    pieces.extend([
        build_header('Как проверить роли любого пользователя', level=3),
        PERMISSIONS_API_TEXT,
    ])

    return '\n'.join(pieces)


def build_page_start():
    url = settings.BACKEND_URL + urlresolvers.reverse(viewname='perms-docs')
    return '\n'.join([
        'Эта секция создана автоматически.',
        'Актуальную версию можно получить по ссылкам: '
        '((%s?format=html html)) '
        '((%s?format=wiki wiki))' % (
            url,
            url,
        ),
    ])


def build_page_end():
    return ''


def build_roles_lists():
    from review.core.logic.assemble import collect_reviews
    from review.core.logic import assemble
    from review.frontend.views import person_reviews
    from review.frontend.views import calibrations_detail_extra
    from review.finance.db.persons import FINANCE_RELATED_ROLES
    from review.api.views.person_reviews import PersonReviewsApiView
    lists = OrderedDict([
        (
            'список ревью',
            collect_reviews.DEFAULT_REVIEW_LIST_ROLES
        ),
        (
            'список ревьюируемых в режиме ревью',
            const.ROLE.PERSON_REVIEW_LIST_RELATED,
        ),
        (
            'история в карточке в режиме ревью',
            person_reviews.PersonReviewsHistoryView.roles,
        ),
        (
            'список калибровок',
            const.ROLE.CALIBRATION.ALL,
        ),
        (
            'история в карточке в режиме калибровки',
            calibrations_detail_extra.CalibrationPersonReviewsHistoryView.roles,
        ),
        (
            'список калибруемых в режиме калибровки',
            const.ROLE.CALIBRATION_PERSON_REVIEWS_ROLES_DEFAULT,
        ),
        (
            'апи c оебс данными для кабинета',
            FINANCE_RELATED_ROLES,
        ),
        (
            'апи c результатами ревью для кабинета',
            PersonReviewsApiView.ROLES,
        ),

    ])

    pieces = [build_header('Где какие роли используются')]
    for list_name, role_list in lists.items():
        pieces.append('* **%s**' % list_name)
        role_list = sorted(role_list, key=roles_sorter)
        for role_id in role_list:
            pieces.append('  * ' + ALL_ROLES_VERBOSE[role_id])
    return '\n'.join(pieces)


def build_header(text, level=3):
    return ('=' * level) + ' ' + text


def build_toc(text, from_h=2, to_h=3):
    return '{{toc text=%s from="h%s" to="h%s"}}' % (
        text,
        from_h,
        to_h,
    )


def build_table_for_roles(role_block_name, roles):
    roles = sorted(roles, key=roles_sorter)
    pieces = [
        build_header(role_block_name, level=4),
        build_table_start(),
        build_roles_perms_table_header(),
        build_table_block(role_block_name=role_block_name, roles=roles),
        build_table_end(),
        build_page_end(),
    ]
    return '\n'.join(pieces)


def build_roles_perms_table_header():
    pieces = [
        build_table_row(
            items=[
                'Пермишны',
                'Роли',
            ],
            is_header=True,
        )
    ]
    return '\n'.join(pieces)


def build_table_start():
    return '++#|'


def build_table_end():
    return '|#++'


def build_table_block(role_block_name, roles):
    roles = sorted(roles, key=roles_sorter)
    pieces = [
        build_table_row(
            items=[
                '',
                role_block_name,
            ],
            is_header=True,
        ),
        build_table_row(
            items=[''] + [
                ALL_ROLES_VERBOSE[role]
                for role in roles
            ],
        )
    ]
    for perm_group_name, perms_group in PERMS_GROUPS.items():
        block_annotations = []
        if perm_group_name not in PERM_GROUP_TO_ROLE_BLOCKS[role_block_name]:
            continue

        pieces.append(
            build_table_row(
                items=[
                    '**%s**' % perm_group_name,
                    PERM_GROUP_BLOCK_COMMENT.get(
                        (role_block_name, perm_group_name),
                        ''
                    ),
                ]
            )
        )

        for perm_id, perm_verbose in perms_group.items():

            values = []
            for role_id in roles:
                value = get_role_value(role_id, perm_id)
                if isinstance(value, str):
                    if value not in block_annotations:
                        block_annotations.append(value)
                    stars = (STAR * (block_annotations.index(value) + 1))
                    values.append(MAYBE_TPL % stars)
                else:
                    values.append(YES if value else NO)

            pieces.append(
                build_table_row(
                    items=[perm_verbose] + values
                )
            )

        if block_annotations:
            annotation_strings = []
            for star_count, key in enumerate(block_annotations, start=1):
                annotation_strings.append(
                    ' '.join([
                        MAYBE_TPL % (STAR * star_count),
                        VERBOSE_DYNAMIC_PERM.get(key, key)
                    ])
                )
            pieces.append(
                build_table_row(
                    items=[' ', '\n'.join(annotation_strings)]
                )
            )

    return '\n'.join(pieces)


def build_table_row(items, is_header=False):
    row = '|| %s ||' % '|'.join(items)
    if is_header:
        row = '**%s**' % row
    return row


def get_role_value(role_id, perm_id):
    value = None
    if perm_id in REVIEW_PERMS.ALL:
        value = get_review_perm_value(role_id, perm_id)
    if perm_id in PERSON_REVIEW_PERMS.ALL:
        value = get_person_review_perm_value(role_id, perm_id)
    if perm_id in GLOBAL_PERMS.ALL:
        value = get_global_perm_value(role_id, perm_id)
    if perm_id in CALIBRATION_PERMS.ALL:
        value = get_calibration_perm_value(role_id, perm_id)
    if perm_id in VERBOSE_PERSON_PERMS:
        value = get_person_perm_value(role_id, perm_id)
    return value


def get_review_perm_value(role_id, perm_id):
    return perm_id in REVIEW_PERMS.ROLE_PERMS.get(role_id, [])


def get_person_review_perm_value(role_id, perm_id):
    perm_set = permissions_person_review.ROLE_PERMS.get(role_id)
    if perm_set is None:
        return False
    if perm_id in perm_set.const_perms:
        return True
    if perm_id in perm_set.dynamic_perms:
        return perm_set.dynamic_perms[perm_id].__name__
    return False


def get_global_perm_value(role_id, perm_id):
    from review.core.logic import permissions
    condition = permissions.GLOBAL_ROLES.PERMS[perm_id]
    if callable(condition):
        return condition.__name__
    else:
        return role_id in condition


def get_calibration_perm_value(role_id, perm_id):
    return perm_id in CALIBRATION_PERMS.ROLE_PERMS.get(role_id, [])


def get_person_perm_value(role_id, perm_id):
    from review.finance.db.persons import FINANCE_RELATED_ROLES
    return role_id in FINANCE_RELATED_ROLES
