# coding: utf-8
"""
Generate data in database for development and testing.
"""

import random
import uuid
import itertools
import operator
import arrow
from decimal import Decimal

from django.utils import timezone
from bulk_update.helper import bulk_update

from review.lib import helpers
from review.lib import datetimes
from review.core import models as core_models
from review.core import const
from review.core.logic import roles
from review.staff import logic as staff_logic
from review.staff import models as staff_models


@helpers.timeit
def tmp_create_all(clean=False):
    tmp_create_reviews(clean=clean)
    tmp_create_person_reviews(clean=clean)
    tmp_create_review_reviewers(clean=clean)
    tmp_create_super_reviewers(clean=clean)
    tmp_create_readers(clean=clean)
    tmp_create_review_changes(clean=clean)
    tmp_create_review_goodies(clean=clean)
    tmp_structure_changes(clean=clean)
    roles.denormalize_person_review_roles()


@helpers.timeit
def tmp_create_reviews(clean=False):
    if clean:
        core_models.Review.objects.filter(migrated=False).delete()

    creator = core_models.GlobalRole.objects.filter(
        type=const.ROLE.GLOBAL.REVIEW_CREATOR,
    ).values_list('person_id', flat=True).first()

    reviews = [
        core_models.Review(
            name='Ревью Инфраструктуры, окт 2016',
            start_date='2016-10-01',
            finish_date='2016-12-01',
            status=const.REVIEW_STATUS.IN_PROGRESS,
            mark_mode=const.REVIEW_MODE.MODE_MANUAL,
            goldstar_mode=const.REVIEW_MODE.MODE_MANUAL,
            level_change_mode=const.REVIEW_MODE.MODE_MANUAL,
            salary_change_mode=const.REVIEW_MODE.MODE_MANUAL,
            bonus_mode=const.REVIEW_MODE.MODE_MANUAL,
            options_rsu_mode=const.REVIEW_MODE.MODE_MANUAL,
            author_id=creator,
        ),
        core_models.Review(
            name='Ревью Инфраструктуры, май 2017',
            start_date='2017-01-04',
            finish_date='2017-01-06',
            status=const.REVIEW_STATUS.IN_PROGRESS,
            mark_mode=const.REVIEW_MODE.MODE_DISABLED,
            level_change_mode=const.REVIEW_MODE.MODE_MANUAL,
            salary_change_mode=const.REVIEW_MODE.MODE_MANUAL,
            bonus_mode=const.REVIEW_MODE.MODE_DISABLED,
            options_rsu_mode=const.REVIEW_MODE.MODE_MANUAL,
            author_id=creator,
        ),
        core_models.Review(
            name='Ревью Поиска, апрель 2017',
            start_date='2017-01-03',
            finish_date='2017-01-05',
            status=const.REVIEW_STATUS.IN_PROGRESS,
            author_id=creator,
        ),
    ]
    core_models.Review.objects.bulk_create(reviews)


@helpers.timeit
def tmp_create_review_goodies(clean=False):
    if clean:
        core_models.Goodie.objects.filter(review__migrated=False).delete()

    goodies = []

    for review in core_models.Review.objects.filter(migrated=False):
        for mark, goldstar in itertools.product(
                list(const.MARK.NON_EMPTY),
                list(const.GOLDSTAR.ALL)
        ):
            goodies.append(
                core_models.Goodie(
                    review=review,
                    mark=mark,
                    goldstar=goldstar,
                    level_change=random.choice([0, 1]),
                    salary_change=Decimal(random.randrange(0, 100, 4)),
                    bonus=Decimal(random.randrange(0, 200, 5)),
                    options_rsu=random.randrange(0, 2000, 100),
                )
            )
    core_models.Goodie.objects.bulk_create(goodies)


@helpers.timeit
def tmp_create_person_reviews(clean=False):
    if clean:
        core_models.PersonReview.objects.filter(review__migrated=False).delete()

    review_to_department = {
        'инфраструктур': 'yandex_infra',
        'поиск': 'yandex_rkub_portal',
    }

    person_reviews = []
    for substr, dep_slug in review_to_department.items():
        department = staff_logic.ensure_department_model(dep_slug)
        head = staff_logic.get_department_head(department)
        persons = staff_logic.get_persons_from_department_tree(root=department)
        dep_reviews = core_models.Review.objects.filter(
            name__icontains=substr,
            migrated=False,
        )
        for review in dep_reviews:
            for person in persons:
                if person == head:
                    continue
                person_reviews.append(
                    core_models.PersonReview(
                        review=review,
                        person=person,
                        updated_at=timezone.now(),
                        **generate_person_review_params()
                    )
                )
    return core_models.PersonReview.objects.bulk_create(person_reviews)


def generate_person_review_params():
    params = {'status': random.choice(list(const.PERSON_REVIEW_STATUS.ALL))}
    if params['status'] == const.PERSON_REVIEW_STATUS.WAIT_EVALUATION:
        params['mark'] = const.MARK.NOT_SET
        params['approve_level'] = 0
    if params['status'] == const.PERSON_REVIEW_STATUS.EVALUATION:
        params['mark'] = random.choice(list(const.MARK.ALL))
        params['approve_level'] = 0
    if params['status'] == const.PERSON_REVIEW_STATUS.APPROVAL:
        params['mark'] = random.choice(list(const.MARK.ALL))
        params['approve_level'] = 1

    if params.get('mark') in const.MARK.NON_EMPTY:
        params['goldstar'] = random.choice(list(const.GOLDSTAR.ALL))
        params['level_change'] = random.choice([0, 1])
        params['salary_change'] = Decimal(random.randrange(0, 100, 4))
        params['bonus'] = Decimal(random.randrange(0, 200, 5))
        params['options_rsu'] = random.randrange(0, 2000, 100)
        params['flagged'] = random.choice([True, False])
    return params


@helpers.timeit
def tmp_create_review_reviewers(clean=False):
    if clean:
        core_models.PersonReviewRole.objects.filter(
            type__in=[
                const.ROLE.PERSON_REVIEW.REVIEWER,
                const.ROLE.PERSON_REVIEW.TOP_REVIEWER,
            ],
            person_review__review__migrated=False,
        ).delete()

    roles_list = []
    update_approve_level = []

    person_heads = _get_subordination_mapping()
    for pr in core_models.PersonReview.objects.filter(review__migrated=False):
        head_ids = person_heads.get(pr.person_id, [])
        head_ids = head_ids[:-1]

        pr_roles = []
        for position, head_id in enumerate(head_ids):
            if position == len(head_ids) - 1:
                type = const.ROLE.PERSON_REVIEW.TOP_REVIEWER
            else:
                type = const.ROLE.PERSON_REVIEW.REVIEWER
            pr_roles.append(
                core_models.PersonReviewRole(
                    person_review=pr,
                    person_id=head_id,
                    type=type,
                    position=position,
                )
            )
            if pr.status == const.PERSON_REVIEW_STATUS.APPROVAL:
                pr.approve_level = random.randint(0, len(pr_roles) - 1)
                update_approve_level.append(pr)
        roles_list.extend(pr_roles)
    create_roles_count = core_models.PersonReviewRole.objects.bulk_create(
        objs=roles_list,
        batch_size=1000
    )
    updated_person_reviews_count = bulk_update(
        objs=update_approve_level,
        update_fields=['approve_level'],
    )

    return {
        'created_roles_count': create_roles_count,
        'updated_person_reviews_count': updated_person_reviews_count,
    }


COMMENTS = [
    """
    Ого-го молодец!
    """,
    """
    Ого-го не молодец!
    """,
    """
    Коммент какой-то
    двухстрочный.
    """,
]


@helpers.timeit
def tmp_create_review_changes(clean=False):
    if clean:
        core_models.PersonReviewChange.objects.filter(
            person_review__review__migrated=False,
        ).delete()

    models = []
    imperator = staff_models.Person.objects.get(login='imperator')
    for pr in core_models.PersonReview.objects.filter(review__migrated=False):
        comment = random.choice(COMMENTS)
        diff = {
            const.FIELDS.MARK: {
                'old': pr.mark,
                'new': random.choice(list(const.MARK.ALL)),
            }
        }
        models.append(
            core_models.PersonReviewChange(
                subject=imperator,
                person_review_id=pr.id,
                diff=diff,
                created_at=datetimes.shifted(pr.updated_at, days=-1)
            )
        )
    core_models.PersonReviewChange.objects.bulk_create(
        objs=models,
    )


@helpers.timeit
def _get_subordination_mapping():
    person_heads = {}
    queryset = staff_models.Subordination.objects.order_by('object', 'position')
    grouper = operator.attrgetter('object_id')
    for person_id, subordinations in itertools.groupby(queryset, key=grouper):
        person_heads[person_id] = [s.subject_id for s in subordinations]
    return person_heads


@helpers.timeit
def tmp_create_readers(clean=False):
    if clean:
        core_models.PersonReviewRole.objects.filter(
            type__in=[
                const.ROLE.PERSON_REVIEW.READER,
                const.ROLE.PERSON_REVIEW.SUPERREADER,
            ],
            person_review__review__migrated=False,
        ).delete()

    roles_list = []

    sunset = staff_models.Person.objects.get(login='sunset')
    volozh = staff_models.Person.objects.get(login='volozh')

    for person_review in _get_some_person_reviews(and_my=True):
        roles_list.append(
            core_models.PersonReviewRole(
                person_review=person_review,
                person=sunset,
                type=const.ROLE.PERSON_REVIEW.READER,
            )
        )
    for person_review in _get_some_person_reviews(and_my=True):
        roles_list.append(
            core_models.PersonReviewRole(
                person_review=person_review,
                person=volozh,
                type=const.ROLE.PERSON_REVIEW.SUPERREADER,
            )
        )

    return core_models.PersonReviewRole.objects.bulk_create(objs=roles_list)


def _get_some_person_reviews(and_my=True):
    person_reviews = list(core_models.PersonReview.objects.filter(
        review__migrated=False
    ).order_by('?')[:10])
    if and_my:
        person_reviews.extend(
            core_models.PersonReview.objects.filter(
                person__login='sibirev',
                review__migrated=False,
            ).all()
        )
    return person_reviews


@helpers.timeit
def tmp_create_super_reviewers(clean=False):
    if clean:
        core_models.ReviewRole.objects.filter(
            type=const.ROLE.REVIEW.SUPERREVIEWER,
            review__migrated=False,
        ).delete()

    roles_list = []

    lili = staff_models.Person.objects.get(login='lili-na')
    infra_reviews = core_models.Review.objects.filter(
        name__icontains='инфраструктур',
        migrated=False,
    )
    search_reviews = core_models.Review.objects.filter(
        name__icontains='поиск',
        migrated=False,
    )
    for review in infra_reviews:
        roles_list.append(
            core_models.ReviewRole(
                review=review,
                person=lili,
                type=const.ROLE.REVIEW.SUPERREVIEWER,
            )
        )
    for review in search_reviews:
        roles_list.append(
            core_models.ReviewRole(
                review=review,
                person=lili,
                type=const.ROLE.REVIEW.SUPERREVIEWER,
            )
        )
    return core_models.ReviewRole.objects.bulk_create(objs=roles_list)


@helpers.timeit
def tmp_structure_changes(clean=False):
    if clean:
        staff_models.StaffStructureChange.objects.all().delete()

    person_to_head = staff_models.Subordination.objects.values('subject_id', 'object_id')
    person_to_heads = {}
    for item in person_to_head:
        person_heads = person_to_heads.setdefault(item['object_id'], [])
        person_heads.append(item['subject_id'])

    def shuffled(list_):
        copy = list_[:]
        random.shuffle(copy)
        return copy

    changed_person_to_heads = {
        key: shuffled(value)
        for key, value in person_to_heads.items()
    }
    dates = [
        datetimes.shifted(dt=arrow.now(), months=-1),
        datetimes.shifted(dt=arrow.now(), months=-2),
    ]
    structure_changes = [
        staff_models.StaffStructureChange.objects.create(
            date=date,
            staff_id=str(uuid.uuid4())[:10],
        )
        for date in dates
    ]

    persons_0 = [
        staff_models.PersonHeads(
            structure_change=structure_changes[0],
            person_id=person_id,
            heads=','.join(map(str, heads)),
        )
        for person_id, heads in person_to_heads.items()
    ]
    persons_1 = [
        staff_models.PersonHeads(
            structure_change=structure_changes[1],
            person_id=person_id,
            heads=','.join(map(str, heads)),
        )
        for person_id, heads in changed_person_to_heads.items()
    ]
    staff_models.PersonHeads.objects.bulk_create(persons_0 + persons_1, batch_size=500)
