# coding: utf-8
import yenv

from django import forms
from django import http

from review.core.logic import roles, assemble
from review.lib import views
from review.lib import errors
from review.lib import wf
from review.shortcuts import const
from review.shortcuts import models
from review.shortcuts import serializers
from review.dev import docs


FIELDS = const.FIELDS


class UserPermissionsDebugView(views.View):
    def process_get(self, auth, data):
        if yenv.type == 'production' and not roles.has_global_roles(
            person=auth.user,
            global_roles=[const.ROLE.GLOBAL.SUPPORT]
        ):
            raise errors.PermissionDenied(
                type='permissions'
            )

        person = models.Person.objects.filter(
            login=data['login']
        ).first()
        if person is None:
            return {}

        reviews = assemble.get_reviews(
            subject=person,
            filters={},
            requested_fields={
                'id',
                'name',
            }
        )
        reviews = [r for r in reviews if r.id == 64]
        person_reviews = assemble.get_person_reviews(
            subject=person,
            fields_requested={
                FIELDS.ROLES,
                FIELDS.PERMISSIONS_WRITE,
                FIELDS.PERMISSIONS_READ,
                FIELDS.ID,
                FIELDS.PERSON_LOGIN,
                FIELDS.REVIEW_ID,
            }
        )
        person_reviews_grouped_by_reviews = {}
        for person_review in person_reviews:
            group = person_reviews_grouped_by_reviews.setdefault(
                person_review.review_id, [])
            group.append(person_review)

        calibrations = assemble.get_calibrations(
            subject=person,
            filters={},
            requested_fields={
                'id',
                'name',
            }
        )
        calibration_person_reviews = assemble.get_calibration_person_reviews(
            subject=person,
            filters_chosen={},
            requested_person_review_fields={
                FIELDS.ID,
                FIELDS.PERSON_LOGIN,
            }
        )
        calibration_person_reviews_grouped_by_calibration = {}
        for cpr in calibration_person_reviews:
            group = calibration_person_reviews_grouped_by_calibration.setdefault(
                cpr.calibration_id, [])
            group.append(cpr)
        return serialize_user_permissions_debug(
            reviews=reviews,
            person_reviews=person_reviews_grouped_by_reviews,
            calibrations=calibrations,
            calibration_person_reviews=calibration_person_reviews_grouped_by_calibration,
        )


def serialize_user_permissions_debug(reviews, person_reviews, calibrations, calibration_person_reviews):
    return {
        'reviews': serializers.ExtendedReviewSerializer.serialize_many(
            objects=reviews,
            fields_requested={
                'id',
                'name',
                'roles',
                'permissions'
            }
        ),
        'calibrations': serializers.ExtendedCalibrationSerializer.serialize_many(
            objects=calibrations,
            fields_requested={
                'id',
                'name',
                'roles',
                'permissions'
            },
            context=dict(language='ru'),
        ),
        'person_reviews_by_review': {
            id: serializers.PersonReviewExtendedSerializer.serialize_many(
                objects=_person_reviews,
                fields_requested={
                    const.FIELDS.ID,
                    const.FIELDS.PERSON_LOGIN,
                    const.FIELDS.REVIEW_NAME,
                    'roles',
                    'permissions',
                }
            )
            for id, _person_reviews in person_reviews.items()
        },
        'calibration_person_reviews_by_calibration': {
            id: serializers.CalibrationPersonReviewSerializer.serialize_many(
                objects=_calibration_person_reviews,
                fields_requested={
                    'id',
                    'roles',
                    'permissions',
                    'person_review.person_login',
                    'person_review.id',
                }
            )
            for id, _calibration_person_reviews
            in list(calibration_person_reviews.items())
        }
    }


class PermissionsDocsView(views.View):

    class PermissionsDocsForm(forms.Form):
        format = forms.ChoiceField(
            choices=[
                (_, _) for _ in [
                    'wiki',
                    'html',
                ]
            ],
            required=False,
        )
    form_cls = PermissionsDocsForm

    def process_get(self, auth, data):
        format = data['format'] or 'html'
        return self.get_response(format=format)

    def get_response(self, format):
        content = docs.build_docs_roles_wiki_markup()
        if format == 'html':
            content_type = 'text/html'
            content = wf.format_markup(content)
        else:
            content_type = 'text/plain'
        content_type += '; charset=utf-8'
        return http.HttpResponse(
            content=content,
            content_type=content_type,
            charset='utf-8',
        )
