# coding=utf-8

from datetime import date

import arrow
from django.db.models import Min, Q

from review.lib import datetimes
from review.core import models as core_models
from review.oebs import logic
from review.oebs import const as oebs_const
from review.core import const as core_const


def fetch(person_ids, fields=None, cached_result=False):
    fields = set(fields) if fields else oebs_const.OEBS_DATA_TYPES
    return _fetch_finance(
        fields,
        person_ids,
        cached_result,
    )


def _fetch_finance(fields, person_ids, cached_result):
    fields = fields & oebs_const.OEBS_DATA_TYPES
    oebs_data = logic.get_finance_data(
        fields=fields,
        cached_result=cached_result,
        person_id__in=person_ids,
    )
    crop_dates = _generate_crop_dates(person_ids)
    return _clean_data(oebs_data, crop_dates)


def _clean_data(data, crop_dates):
    for person_id, person_data in data.items():
        crop_date = crop_dates.get(person_id)
        _drop_future_for_date_ranged_events(crop_date, person_data)
        _drop_future_for_date_events(crop_date, person_data)
        _clean_current_salary(person_data)
    return data


def _is_date_from_future(crop_date, date_str):
    if crop_date is None:
        return True
    return date_str > crop_date.isoformat()


def _drop_future_for_date_ranged_events(crop_date, user_data):
    date_ranged_fields = user_data.keys() & oebs_const.DATE_RANGED_EVENT_FIELDS
    for field_name in date_ranged_fields:
        field = user_data[field_name]
        filtered = []
        for item in field:
            date_to = item.get('dateTo')
            date_from = item.get('dateFrom')

            if date_from and _is_date_from_future(crop_date, date_from):
                continue

            if date_to and _is_date_from_future(crop_date, date_to):
                item = dict(item)
                item['dateTo'] = crop_date

            filtered.append(item)
        user_data[field_name] = filtered


def _drop_future_for_date_events(first_day_of_future, user_data):
    date_field_map = oebs_const.DATE_EVENT_FIELDS
    date_fields = user_data.keys() & date_field_map.keys()
    for field_name in date_fields:
        field = user_data[field_name]
        date_field = date_field_map[field_name]
        filtered = []
        for item in field:
            item_date = item.get(date_field)
            if item_date and _is_date_from_future(first_day_of_future, item_date):
                continue
            filtered.append(item)
        user_data[field_name] = filtered


def _clean_current_salary(user_data):
    salary_fields = user_data.keys() & {oebs_const.CURRENT_SALARY}
    for salary_field in salary_fields:
        field = dict(user_data[salary_field])
        user_data[salary_field] = {
            key: value for key, value in field.items()
            if key == 'factFTE'
        }


def _generate_crop_dates(person_ids):
    """
    STAFF-7647 & CIA-237
    Для каждого сотрудника определяем дату, после которой
    финансовые события по этому сотруднику недоступны в апи.

    Для тех, у кого есть активные ревью — обрезаем историю до даты начала.
    (если несколько активных ревью — берем минимальную дату начала)
    Для тех, у кого нет ревью — это последний день предыдущего месяца.
    """
    person_ids = set(person_ids)
    today = arrow.now()

    status_not_announced = ~Q(status=core_const.PERSON_REVIEW_STATUS.ANNOUNCED)

    crop_dates_by_review = dict(
        core_models.PersonReview.objects.filter(
            status_not_announced,
            person_id__in=person_ids,
            review__status__in=core_const.REVIEW_STATUS.ACTIVE,
            review__start_date__lte=today.date()
        ).values_list(
            'person_id'
        ).annotate(
            Min('review__start_date')
        )
    )
    first_day_of_this_month = date(
        year=today.year,
        month=today.month,
        day=1,
    )

    result = {}
    for person_id in person_ids:
        if person_id in crop_dates_by_review:
            first_day_of_future = crop_dates_by_review[person_id]
        else:
            first_day_of_future = first_day_of_this_month
        result[person_id] = datetimes.shifted(first_day_of_future, days=-1)
    return result

# TODO: oebs не может зависеть от core — в реальности обрезание нужно
# только для api и должно быть где-то там
