from collections import defaultdict
from typing import Dict, List, Tuple, Set

from django.db.models import Q

from review.core import const
from review.staff import const as staff_const
from review.staff import models as staff_models
from review.staff import logic as staff_logic

from review.core.logic.roles import get_existing_global_roles


FINANCE_GLOBAL_ROLES = {
    const.ROLE.GLOBAL.LOAN_VIEWER,
}


FINANCE_RELATED_ROLES = {
    staff_const.STAFF_ROLE.DEPARTMENT.HEAD,
    const.ROLE.PERSON.SELF,
    staff_const.STAFF_ROLE.HR.HR_ANALYST,
    staff_const.STAFF_ROLE.HR.HR_PARTNER,
    staff_const.STAFF_ROLE.HR.FINANCE_VIEWER,
} | FINANCE_GLOBAL_ROLES


def get_id_login_roles(subject, filters=None, role_types=None):
    # type: (staff_models.Person, Dict, List) -> List[List[int, str, Set[str]]]
    grouped = get_allowed_persons_grouped(subject, filters, role_types)
    id_to_roles = defaultdict(set)
    id_to_login = {}
    for role_type, person_group in grouped.items():
        for person in person_group:
            id_ = person['id']
            id_to_roles[id_].add(role_type)
            id_to_login[id_] = person['login']
    return [
        [id_, id_to_login[id_], roles]
        for id_, roles in id_to_roles.items()
    ]


def get_allowed_persons_grouped(subject, filters=None, role_types=None):
    role_types = role_types and set(role_types) or FINANCE_RELATED_ROLES
    filters = filters or {}

    grouped_by_role = {role: [] for role in role_types}

    if staff_const.STAFF_ROLE.DEPARTMENT.HEAD in role_types:
        grouped_by_role.update(
            _get_persons_for_head(
                subject=subject,
                filters=filters,
            )
        )
    if const.ROLE.PERSON.SELF in role_types:
        grouped_by_role.update(
            _get_persons_for_self(
                subject=subject,
                filters=filters,
            )
        )
    hr_roles = role_types & {
        staff_const.STAFF_ROLE.HR.HR_PARTNER,
        staff_const.STAFF_ROLE.HR.HR_ANALYST,
        staff_const.STAFF_ROLE.HR.FINANCE_VIEWER,
    }
    if hr_roles:
        grouped_by_role.update(
            _get_persons_for_hrs(
                subject=subject,
                filters=filters,
                roles=hr_roles,
            )
        )
    if FINANCE_GLOBAL_ROLES & role_types:
        grouped_by_role.update(
            _get_persons_for_globals(
                subject=subject,
                filters=filters,
            )
        )
    return grouped_by_role


def _get_persons_for_head(subject, filters):
    subject = staff_logic.ensure_person_model(subject)
    filter_q = _prepare_persons_filter_q(filters)
    subordinate_persons_q = staff_logic.get_all_subordinate_persons_q(
        subject,
        prefix='',
    )
    persons = staff_models.Person.objects.filter(
        subordinate_persons_q,
        filter_q,
    ).values('id', 'login')
    return {staff_const.STAFF_ROLE.DEPARTMENT.HEAD: persons}


def _get_persons_for_self(subject, filters):
    subject = staff_logic.ensure_person_model(subject)
    ids = filters.get('ids', set())
    logins = filters.get('logins', set())
    objects = filters.get('persons')
    if objects:
        ids |= {p.id for p in objects}
    if subject.id in ids or subject.login in logins:
        return {const.ROLE.PERSON.SELF: [
            dict(
                id=subject.id,
                login=subject.login,
            )
        ]}
    else:
        return {const.ROLE.PERSON.SELF: []}


def _get_persons_for_globals(subject, filters):
    subject = staff_logic.ensure_person_model(subject)
    existing_roles = get_existing_global_roles(subject, FINANCE_GLOBAL_ROLES)
    res = {role: [] for role in FINANCE_GLOBAL_ROLES}
    if not existing_roles:
        return res

    filter_q = _prepare_persons_filter_q(filters)
    persons = staff_models.Person.objects.filter(
        filter_q,
    ).values('id', 'login')
    for person in persons:
        for role in existing_roles:
            res[role].append(person)
    return res


def _get_persons_for_hrs(subject, filters, roles):
    subject = staff_logic.ensure_person_model(subject)
    filter_q = _prepare_persons_filter_q(filters)
    persons = staff_models.Person.objects.filter(
        filter_q,
        hr__hr_person=subject,
        hr__type__in=roles,
    ).values('id', 'login', 'hr__type')
    grouped_by_role = defaultdict(list)
    for person in persons:
        grouped_by_role[person.pop('hr__type')].append(person)
    return grouped_by_role


def _prepare_persons_filter_q(filters):
    query_params = {}
    ids = filters.get('ids')
    persons = filters.get('persons')
    logins = filters.get('logins')
    if ids is not None:
        query_params['id__in'] = ids
    if logins is not None:
        query_params['login__in'] = logins
    if persons is not None:
        query_params['id__in'] = [p.id for p in persons]

    return Q(**query_params)
