# coding: utf-8

import decimal
import enum
import logging
from typing import Dict, List, Optional, Tuple

import attr
import waffle

from review.bi import logic as bi_logic
from review.core import const
from review.oebs import loan_api


log = logging.getLogger(__name__)

CONSIDERED_MARKS_COUNT = 2
RESTRICTED_DEP_URLS = {
    'yandex_personal_vertserv',  # Вертикали
}
OLD_MARKS = ('B', 'C-', 'C', 'D', 'E')
MARKS = (
    'poor',  # -
    'below',  # +-
    'good',  # +
    'great',  # ++
    'outstanding',  # +++
    'amazing',  # ++++
)
NEW_SCALES_FORMAT_STARTS_AT = 71

# There are different grade restrictions for different loan types
HIGH_GRADE_FOR_LONG_LOAN = 17
HIGH_GRADE_FOR_SHORT_LOAN = 16
LOW_GRADE_FOR_LONG_LOAN = 15
LOW_GRADE_FOR_SHORT_LOAN = 14


def get_accepted_marks(low, old):
    # type: (bool, bool) -> List[str]
    if old:
        return OLD_MARKS[2:]
    elif low:
        return MARKS[3:]
    else:
        return MARKS[2:]


def replace_cyrillic(mark):
    if mark == 'С':
        return 'C'
    return mark


def check_mark(grade, mark, low_grade, high_grade):
    # type: (...) -> Tuple[Optional[int], Optional[str]]
    mark = replace_cyrillic(mark)
    accepted_marks = get_accepted_marks(low=grade < high_grade, old=mark in OLD_MARKS)
    accepted_grade = high_grade if grade >= high_grade else low_grade
    required_mark = None if mark in accepted_marks and grade >= accepted_grade else accepted_marks[0]
    required_grade = max(grade, low_grade) if required_mark else None
    return required_grade, required_mark


@attr.s
class MarkInfo(object):
    mark = attr.ib(type=str, default=const.MARK.NO_MARK)
    scale_id = attr.ib(type=int, default=NEW_SCALES_FORMAT_STARTS_AT)
    grade = attr.ib(type=int, default=0)


History = Tuple[Optional[MarkInfo], Optional[MarkInfo]]


def get_missing_loan_requirements(
    last_marks,  # type: List[MarkInfo]
    low_grade,  # type: int
    high_grade,  # type: int
):
    # type: (...) -> History
    missing_requirements = []

    while len(last_marks) < CONSIDERED_MARKS_COUNT:
        last_marks = [MarkInfo()] + last_marks

    for mark_info in last_marks:
        required_grade, required_mark = check_mark(
            mark_info.grade,
            mark_info.mark,
            low_grade,
            high_grade,
        )
        if required_grade or required_mark:
            scale_to_answer = mark_info.scale_id
            if mark_info.scale_id < NEW_SCALES_FORMAT_STARTS_AT:
                # looks like awful hack
                # old scales format differs from current
                # suspec that frontend goes to review-api for marks-scales
                # and do not want scales with old format
                scale_to_answer = None
            missing_requirements.append(MarkInfo(
                grade=required_grade,
                mark=required_mark,
                scale_id=scale_to_answer,
            ))
        else:
            missing_requirements.append(None)

    return missing_requirements


def get_last_marks(review_bi_income):
    # type: (bi_logic.PersonIncome) -> List[MarkInfo]
    res = []
    if review_bi_income.mark_last is not None:
        res.append(MarkInfo(
            mark=review_bi_income.mark_last.mark,
            scale_id=review_bi_income.mark_last.scale_id,
            grade=review_bi_income.grade_last,
        ))
    if review_bi_income.mark_current is not None:
        res.append(MarkInfo(
            mark=review_bi_income.mark_current.mark,
            scale_id=review_bi_income.mark_current.scale_id,
            grade=review_bi_income.grade_current,
        ))
    return res


oebs_requirements = [
    'is_resident',
    'is_not_on_maternity_leave',
    'has_permanent_job_contract',
    'has_been_working_one_year',
    'does_not_have_loan',
]


review_requirements = [
    'has_review_marks_short',
    'has_review_marks_long',
]


options_requirements = [
    'has_not_vested_options',
]


class SALARY_TYPE(enum.Enum):
    AVERAGE = 1
    INCOME_ONLY = 2


@attr.s
class AvailableSums:
    max_monthly_payment = attr.ib(type=decimal.Decimal)
    max_loan_amount = attr.ib(type=decimal.Decimal)

    def as_dict(self):
        return {
            k: str(v)
            for k, v in attr.asdict(self).items()
        }


@attr.s
class LoanReqs(object):
    @attr.s
    class WorkExp(object):
        years = attr.ib(type=int, default=0)
        months = attr.ib(type=int, default=0)
        days = attr.ib(type=int, default=0)

    @attr.s
    class BoolReqs(object):
        is_resident = attr.ib(type=bool, default=None)
        is_not_on_maternity_leave = attr.ib(type=bool, default=None)
        has_permanent_job_contract = attr.ib(type=bool, default=None)
        has_been_working_one_year = attr.ib(type=bool, default=None)
        does_not_have_loan = attr.ib(type=bool, default=None)
        has_review_marks_short = attr.ib(type=bool, default=None)
        has_review_marks_long = attr.ib(type=bool, default=None)
        has_not_vested_options = attr.ib(type=bool, default=None)

        def clear(self, requirements):
            # type: (List[str]) -> None
            for field in requirements:
                setattr(self, field, None)

        def get_values(self, fields):
            # type: (List[str]) -> List[Optional[bool]]
            return [getattr(self, field) for field in fields]

    requirements = attr.ib(type=BoolReqs, default=None)
    work_experience = attr.ib(type=WorkExp, default=None)
    is_short_loan_available = attr.ib(type=bool, default=None)
    is_long_loan_available = attr.ib(type=bool, default=None)
    short_loan_missing_requirements = attr.ib(type=History, default=None)
    long_loan_missing_requirements = attr.ib(type=History, default=None)
    last_reviews_results = attr.ib(type=List[MarkInfo], default=None)
    not_vested = attr.ib(type=Optional[List], default=None)
    hold_amount = attr.ib(type=int, default=None)
    exec_doc = attr.ib(type=int, default=None)
    skip_grade_checks = attr.ib(type=bool, default=None)
    has_errors = attr.ib(type=bool, default=None)
    errors = attr.ib(type=Dict[str, str], factory=dict)

    available_sums = attr.ib(
        type=Optional[Dict[SALARY_TYPE, AvailableSums]],
        default=None,
    )

    def set_error(self, fields, error):
        # type: (List[str], str) -> None
        for field in fields:
            self.errors[field] = error

    def as_dict(self):
        res = attr.asdict(self)
        if self.available_sums is not None:
            res['available_sums'] = {
                k.name: v.as_dict()
                for k, v in self.available_sums.items()
            }
        return res


def check_loan_req(
    oebs_loan,  # type: Optional[loan_api.OebsLoan]
    has_not_vested,  # type: Optional[bool]
    not_vested,  # type: Optional[List]
    data_request_errors,  # type: Dict
    person_income,  # type: Optional[bi_logic.PersonIncome]
    person_department_urls=None,  # type: List[str]
    is_viewing_self=True,  # type: bool
    period=None,  # type: Optional[int]
):
    # type: (...) -> LoanReqs
    person_department_urls = person_department_urls or []

    requirements = LoanReqs.BoolReqs()
    result = LoanReqs(requirements=requirements)

    # requirements based on OEBS response
    try:
        if oebs_loan is None:
            error = data_request_errors['oebs_loan']
            result.set_error(oebs_requirements, error)
        elif not oebs_loan.logins[0].hiddenInfo:
            person_loan = oebs_loan.logins[0]
            requirements.is_resident = person_loan.accountNotResident == 'N'
            requirements.is_not_on_maternity_leave = person_loan.maternityLeave == 'N'
            requirements.has_permanent_job_contract = person_loan.temporaryContract == 'N'
            if not isinstance(person_loan.expYears, int):
                raise ValueError(
                    'expYears have to be int, current type is {}'.format(type(int))
                )
            requirements.has_been_working_one_year = person_loan.expYears > 0
            requirements.does_not_have_loan = person_loan.loanFlag == 'N'
            result.work_experience = LoanReqs.WorkExp(
                years=person_loan.expYears,
                months=person_loan.expMonths,
                days=person_loan.expDays,
            )
            result.hold_amount = person_loan.holdAmount
            result.exec_doc = person_loan.execDoc
        else:
            error = 'OEBS error: hidden info'
            result.set_error(oebs_requirements, error)
        if oebs_loan and period is not None:
            result.available_sums = get_every_availbale_sum(
                oebs_loan,
                period,
            )
    except Exception:
        log.warning('OEBS wrong data format', exc_info=True)
        error = 'OEBS error: wrong data format'
        result.set_error(oebs_requirements, error)
        requirements.clear(oebs_requirements)

    # requirements based on REVIEW response
    result.skip_grade_checks = bool(
        set(person_department_urls) & RESTRICTED_DEP_URLS
    )
    if result.skip_grade_checks:
        short_loan_requirements = oebs_requirements
        all_requirements = oebs_requirements + options_requirements
    else:
        short_loan_requirements = oebs_requirements + ['has_review_marks_short']
        all_requirements = oebs_requirements + review_requirements + options_requirements

    try:
        if person_income is None:
            error = data_request_errors['income']
            result.set_error(review_requirements, error)
        else:
            last_marks = get_last_marks(person_income)
            show_marks = (
                is_viewing_self or
                bi_logic.grade_lte_18(person_income.grade_main)
            )
            if show_marks:
                result.last_reviews_results = last_marks

            if not result.skip_grade_checks:
                short_missing = get_missing_loan_requirements(
                    last_marks,
                    low_grade=LOW_GRADE_FOR_SHORT_LOAN,
                    high_grade=HIGH_GRADE_FOR_SHORT_LOAN,
                )
                long_missing = get_missing_loan_requirements(
                    last_marks,
                    low_grade=LOW_GRADE_FOR_LONG_LOAN,
                    high_grade=HIGH_GRADE_FOR_LONG_LOAN,
                )
                result.short_loan_missing_requirements = short_missing
                result.long_loan_missing_requirements = long_missing
                requirements.has_review_marks_short = not any(short_missing)
                requirements.has_review_marks_long = not any(long_missing)
    except Exception:
        error = 'REVIEW error: wrong data format'
        result.set_error(review_requirements, error)
        requirements.clear(review_requirements)

    # requirements based on REVIEW response
    try:
        if data_request_errors.get('finance') is not None:
            error = data_request_errors['finance']
            result.set_error(options_requirements, error)
        else:
            requirements.has_not_vested_options = has_not_vested
            result.not_vested = not_vested
    except Exception:
        error = 'REVIEW error: wrong data format'
        result.set_error(options_requirements, error)
        requirements.clear(options_requirements)

    # short loan
    short_loan_requirements_values = requirements.get_values(short_loan_requirements)
    if None not in short_loan_requirements_values:
        result.is_short_loan_available = all(short_loan_requirements_values)

    # long loan
    all_requirements_values = requirements.get_values(all_requirements)
    if None not in all_requirements_values:
        result.is_long_loan_available = all(all_requirements_values)

    result.has_errors = None in all_requirements_values

    return result


def _get_profit(refinance_rate):
    return refinance_rate * 2 / 3


def _get_max_monthly_payment(salary, ndfl, hold_rate, hold_amount):
    return (salary * (1 - ndfl / 100) - hold_amount) * hold_rate / 100


def _get_max_loan_amount(monthly_payment, tax_ratio, annuity_ratio):
    return monthly_payment / (tax_ratio + annuity_ratio)


def _get_tax_ratio(refinance_rate, annual_rate, mat_profit_ndfl):
    return (_get_profit(refinance_rate) - annual_rate / 100) * mat_profit_ndfl / 100 * 30 / 365


def _get_annuity_ratio_short(period):
    return decimal.Decimal(1 / period)


def _get_annuity_ratio_long(period, annual_rate):
    monthly_rate = annual_rate / 12
    return decimal.Decimal(monthly_rate / 100 / (1 - pow(1 + monthly_rate / 100, -period)))


def _as_money(val):
    return decimal.Decimal(val).quantize(decimal.Decimal('1.00'))


def _get_availbale_sums(
    salary_type: SALARY_TYPE,
    oebs_loan: Optional[loan_api.OebsLoan],
    period: int,
) -> AvailableSums:
    person_loan = oebs_loan.logins[0]
    three_years = 3 * 12
    if period <= three_years:
        annual_rate = 0
        annuity_ratio = _get_annuity_ratio_short(period)
    else:
        annual_rate = 3
        annuity_ratio = _get_annuity_ratio_long(period, annual_rate)

    if waffle.switch_is_active('need_pay_taxes_for_matprofit'):
        tax_ratio = _get_tax_ratio(
            decimal.Decimal(oebs_loan.refinanceRate),
            decimal.Decimal(person_loan.matProfitNDFL),
            annual_rate,
        )
    else:
        tax_ratio = 0

    if salary_type == SALARY_TYPE.AVERAGE:
        salary = person_loan.avgIncome
        hold_rate = 40
    else:
        salary = person_loan.salarySum
        hold_rate = 45
    salary = _as_money(salary)

    max_monthly_payment = _get_max_monthly_payment(
        salary,
        decimal.Decimal(person_loan.NDFL),
        hold_rate,
        decimal.Decimal(person_loan.holdAmount),
    )
    max_loan_amount = _get_max_loan_amount(
        max_monthly_payment,
        tax_ratio,
        annuity_ratio,
    )
    return AvailableSums(
        _as_money(max_monthly_payment),
        _as_money(max_loan_amount),
    )


def get_every_availbale_sum(
    oebs_loan: Optional[loan_api.OebsLoan],
    period: int,
) -> Dict[SALARY_TYPE, AvailableSums]:
    return {
        salary_t: _get_availbale_sums(
            salary_t,
            oebs_loan,
            period,
        )
        for salary_t in SALARY_TYPE
    }
