# coding: utf-8
from __future__ import unicode_literals

import logging
from typing import Dict, List, Optional, Tuple

from waffle import switch_is_active

from review.bi import logic as bi_logic
from review.bi.person_income import PersonIncome
from review.core import const
from review.core.logic import roles
from review.finance import (
    const as finance_const,
    db,
    loan,
    permissions,
    person,
)
from review.lib import errors
from review.oebs import loan_api
from review.staff import models as staff_models


log = logging.getLogger(__name__)


def get_persons(subject, filters, cached_result=True):
    # type: (staff_models.Person, Dict, bool) -> List[person.Person]
    id_login_roles = db.persons.get_id_login_roles(
        subject=subject,
        filters=filters,
    )
    logins, person_ids = [], set()
    for person_id, login, _ in id_login_roles:
        logins.append(login)
        person_ids.add(person_id)

    # finance data is fetched from two sources: oebs and bi
    # bi has more relevant data, and for new fields is better to use
    finance_data = db.finance.fetch(
        person_ids=person_ids,
        cached_result=cached_result,
    )
    grades = bi_logic.get_actual_grades(logins)
    rates = bi_logic.get_persons_rates(
        person_ids,
        [finance_const.RUB, finance_const.USD],
    )

    persons = []
    for person_id, login, roles in id_login_roles:
        grade = grades.get(person_id) or None
        perms = permissions.get_for_person(
            subject=subject,
            roles=roles,
            observable_login=login,
            grade=grade,
        )
        prsn = person.create(
            id_=person_id,
            login=login,
            permissions=perms,
            finance=finance_data.get(person_id) or {},
            grade=grade,
            rates=rates[person_id],
        )
        persons.append(prsn)

    if 'persons' in filters:
        unavailable = [
            it for it in filters['persons']
            if it.id not in person_ids
        ]
        for person_model in unavailable:
            prsn = person.create(
                person_model.id,
                person_model.login,
                permissions=permissions.ForPerson(),
                finance={},
                grade=None,
                rates=rates[None],
            )
            persons.append(prsn)

    return persons


def get_loan_info(
    subject,  # type: staff_models.Person
):
    # type: (...) -> loan_api.OebsLoan
    # some info from OEBS is not actual, so we use bi data instead
    try:
        loan_data = loan_api.get_person_loan(subject.login)
    except Exception as e:
        log.warning(
            'Can\'t fetch data for {}'.format(subject.login),
            exc_info=True,
        )
        raise errors.Error('Fail to fetch data from oebs')
    if not loan_data.logins:
        raise ValueError(f'Logins oebs field is empty for {subject.login}')
    income = bi_logic.get_person_income(subject.login)
    if not income:
        raise errors.NotFound(type='BiPersonIncome', id=subject.login)

    loan_data.logins[0].avgIncome = income.avg_salary
    loan_data.logins[0].salarySum = income.salary
    return loan_data


def get_loan_requirements(
    subject,  # type: staff_models.Person
    person,  # type: staff_models.Person
    period,  # type: Optional[int]
):
    # type: (...) -> loan.LoanReqs
    is_loan_viewer = roles.has_global_roles(
        subject,
        [const.ROLE.GLOBAL.LOAN_VIEWER],
    )
    is_viewing_self = subject.id == person.id
    if not (is_loan_viewer or is_viewing_self):
        raise errors.PermissionDenied(
            type='loan_requirements',
            id=person.login,
        )
    login = person.login
    data_request_errors = {}
    has_not_vested = None
    not_vested = None
    person_income = None
    oebs_loan = None

    persons = get_persons(subject, filters={'logins': [login]})
    if not persons:
        log.exception('Couldn\'t find person'.format(login))
        data_request_errors['finance'] = 'REVIEW error: no finance for {}'.format(login)

    try:
        oebs_loan = get_loan_info(person)
    except Exception as err:
        log.warning('Fail to get loan for {}'.format(login))
        data_request_errors['oebs_loan'] = 'OEBS error: {}'.format(err)

    person_income = bi_logic.get_person_income(login)
    if not person_income:
        log.warning('No income data for {}'.format(login))
        data_request_errors['person_income'] = 'REVIEW error: no income for {}'.format(login)

    if persons:
        has_not_vested, not_vested = get_not_vested_options_info(persons[0], person_income)

    departments_urls = (
        person.department.get_ancestors()
        .values_list('slug', flat=True)
    )

    return loan.check_loan_req(
        oebs_loan=oebs_loan,
        has_not_vested=has_not_vested,
        not_vested=not_vested,
        data_request_errors=data_request_errors,
        person_income=person_income,
        person_department_urls=departments_urls,
        is_viewing_self=is_viewing_self,
        period=period,
    )


def get_not_vested_options_info(
    person: person.Person,
    person_income: PersonIncome
) -> Tuple[bool, List[person.NotVested]]:
    if not switch_is_active('use_deferred_payments_as_not_vested_options_for_loan'):
        return person.has_not_vested, person.not_vested

    permissions = person.permissions_for_observer
    has_not_vested = person.has_not_vested
    payments_as_not_vested_options = get_deferred_payments_as_not_vested_options(person_income)

    if permissions.has_not_vested:
        has_not_vested |= bool(payments_as_not_vested_options)

    if permissions.not_vested:
        not_vested_options = (person.not_vested or []) + payments_as_not_vested_options
    else:
        not_vested_options = None

    return has_not_vested, not_vested_options


def get_deferred_payments_as_not_vested_options(person_income: PersonIncome) -> List[person.NotVested]:
    not_paid_payments_by_type = {}
    if not person_income:
        return []

    for bi_income in person_income.income.from_current_date:
        if not bi_income.forecast:
            continue

        for field in ('deferred_payment', 'vesting_cash_payment'):
            payment_amount = getattr(bi_income, field)
            if payment_amount > 0:
                payment = not_paid_payments_by_type.setdefault(field, person.NotVested(type=field))
                payment.amount += payment_amount
                payment.in_rub = payment.in_usd = None

    return list(not_paid_payments_by_type.values())
