import datetime
import logging
from decimal import Decimal
from typing import Dict, List, Optional

import attr


log = logging.getLogger(__name__)


@attr.s(
    slots=True,
    hash=False,
    repr=False,
)
class Grant(object):
    """
    Now represents only subset of data from OEBS.
    OEBS returns far more rich data, but for now - we need only these fields
    """

    @attr.s(slots=True, hash=False, repr=False)
    class VestingInfo(object):
        amount = attr.ib(type=int)
        date = attr.ib(type=datetime.datetime)

    id = attr.ib(type=int)
    type = attr.ib(type=str)
    exercisable_units = attr.ib(type=Optional[int])
    current_sell_price = attr.ib(type=Decimal)
    price_for_receiver = attr.ib(type=Decimal)
    currency = attr.ib(type=str)
    vesting_schedule = attr.ib(type=List[VestingInfo])


def _from_oebs_dict(oebs_dict):
    # type: (Dict) -> Grant
    vesting_data = oebs_dict.get('vesting') or []
    not_dict_vesting = next(
        (it for it in vesting_data if not isinstance(it, dict)),
        None
    )
    if not_dict_vesting:
        raise TypeError(
            'Expecting dict for vesting, got {}'.format(type(not_dict_vesting))
        )

    return Grant(
        id=int(oebs_dict['grantId']),
        type=str(oebs_dict['classCode']),
        exercisable_units=oebs_dict['grantExercisableUnits'],
        current_sell_price=Decimal(oebs_dict['shareCostSysdate']),
        price_for_receiver=Decimal(oebs_dict['grantPrice']),
        currency=oebs_dict['grantCurrency'],
        vesting_schedule=[
            Grant.VestingInfo(
                amount=int(it['vestAmount']),
                date=datetime.datetime.strptime(it['vestDate'], "%Y-%m-%d").date(),
            )
            for it in vesting_data
            if int(it['vestAmount']) > 0
        ]
    )


def parse_options_history(history, converting_for=None):
    # type: (List[Dict], str) -> List[Grant]
    if history is None:
        return []
    if not isinstance(history, list):
        log.warning(
            'Expecting list for history, got {} for {}'.format(
                type(history),
                converting_for,
            )
        )
        return []
    res = []
    for it in history:
        if not isinstance(it, dict):
            log.warning(
                'Expecting dict for grant, got {} for {}'.format(
                    type(it),
                    converting_for,
                )
            )
            continue
        try:
            grant = _from_oebs_dict(it)
            res.append(grant)
        except Exception:
            log.warning(
                'Fail to convert data for {}'.format(converting_for),
                exc_info=True,
            )
    return res
