import datetime
import logging
from decimal import Decimal
from typing import Dict, List, Optional

import datetime as dt
import logging
from copy import deepcopy
from decimal import Decimal
from typing import Any, DefaultDict, Dict, List, Optional

from review.finance.permissions import ForPerson

from review.bi import logic as bi_logic
from review.core import const
from review.finance import (
    const as finance_const,
    options,
    permissions,
)
from review.oebs import (
    const as oebs_const,
)
import attr

from review.core import const


log = logging.getLogger(__name__)


@attr.s(slots=True, repr=False)
class NotVested(object):
    type = attr.ib(type=str)
    amount = attr.ib(type=int, default=0)
    in_usd = attr.ib(type=Optional[int], default=0)
    in_rub = attr.ib(type=Optional[int], default=0)

    def __repr__(self):
        return 'NotVested {}'.format(self.type)


@attr.s(
    slots=True,
    hash=False,
    repr=False
)
class Person(object):

    id = attr.ib(default=const.NOT_SET)
    login = attr.ib(default=const.NOT_SET)

    # from oebs
    salary_history = attr.ib(default=None)
    grade_history = attr.ib(default=None)
    bonus_history = attr.ib(default=None)
    options_history = attr.ib(default=None)
    current_salary = attr.ib(default=None)
    current_loans = attr.ib(default=None)
    social_package = attr.ib(default=None)

    # from bi
    grade = attr.ib(type=Optional[int], default=None)
    to_usd = attr.ib(type=Optional[Decimal], default=None)
    to_rub = attr.ib(type=Optional[Decimal], default=None)

    # calculable
    not_vested = attr.ib(default=None, type=Optional[List[NotVested]])
    has_not_vested = attr.ib(default=None, type=Optional[bool])

    permissions_for_observer = attr.ib(default=None, type=ForPerson)

    data_fields = (
        'salary_history',
        'grade_history',
        'bonus_history',
        'options_history',
        'current_salary',
        'current_loans',
        'social_package',
        'not_vested',
        'has_not_vested',
    )

    def __hash__(self):
        return self.id

    def __repr__(self):
        return '<RE: %s>' % (
            self.login if self.login is not const.NOT_SET
            else self.id,
        )


def create(
    id_: int,
    login: str,
    permissions: permissions.ForPerson,
    finance: Dict[str, Any],
    grade: Optional[int],
    rates: DefaultDict[str, DefaultDict[str, Decimal]],
) -> Person:
    # Code in this file assumes values inside finance are Union[None, List, Dict]
    # But this is just as assumption, due to unstable oebs api
    person = Person(id_, login)
    person.permissions_for_observer = permissions
    if permissions.oebs_data:
        person.salary_history = finance.get(oebs_const.SALARY_HISTORY)
        person.bonus_history = finance.get(oebs_const.BONUS_HISTORY)
        person.current_salary = finance.get(oebs_const.CURRENT_SALARY)
        person.current_loans = finance.get(oebs_const.CURRENT_LOANS)
        person.social_package = finance.get(oebs_const.SOCIAL_PACKAGE)
        person.grade_history = _get_grade_history(
            finance.get(oebs_const.GRADE_HISTORY),
        )
        person.grade = grade
    else:
        # legacy
        # fields from oebs can be string _NO_ACCESS_
        person.salary_history = const.NO_ACCESS
        person.bonus_history = const.NO_ACCESS
        person.current_salary = const.NO_ACCESS
        person.current_loans = const.NO_ACCESS
        person.social_package = const.NO_ACCESS
        person.grade_history = const.NO_ACCESS

    options_history = _get_options_history(
        finance.get(oebs_const.OPTION_HISTORY),
        permissions.show_excercisable,
    )
    if permissions.options_history:
        person.options_history = options_history
    else:
        person.options_history = const.NO_ACCESS

    to_usd = rates[finance_const.USD][finance_const.RUB]
    to_rub = rates[finance_const.RUB][finance_const.USD]

    if to_usd:
        person.to_usd = to_usd
    if to_rub:
        person.to_rub = to_rub

    parsed_options = options.parse_options_history(options_history, login)
    not_vested = _get_not_vested_options(
        parsed_options,
        to_usd,
        to_rub,
    )

    if permissions.has_not_vested:
        person.has_not_vested = bool(not_vested)

    if permissions.not_vested:
        if not_vested and (to_rub == 0 or to_usd == 0):
            log.warning(
                'Conversion rates for {} are screwed. To usd {} to rub {}'
                .format(login, to_usd, to_rub)
            )
        person.not_vested = not_vested

    return person


def _get_options_history(options_from_finance, show_excercisable):
    # type: (Optional[List[Dict]], bool) -> Optional[List[Dict]]

    if not isinstance(options_from_finance, list):
        return options_from_finance
    if show_excercisable:
        return options_from_finance

    options_history = deepcopy(options_from_finance)
    for option in options_history:
        if isinstance(option, dict):
            option['grantExercisableUnits'] = 0

    return options_history


def _get_grade_history(grades_from_finance):
    # type: (Optional[List[Dict]]) -> Optional[List[Dict]]
    if not isinstance(grades_from_finance, list):
        return grades_from_finance
    grade_history = deepcopy(grades_from_finance)
    for grade in grade_history:
        if isinstance(grade, dict):
            for field in ('min', 'mid', 'max'):
                grade.pop(field, None)
    return grade_history


def _get_not_vested_options(
    parsed_history,  # type: List[options.Grant]
    to_usd,  # type: Decimal
    to_rub,  # type: Decimal
):
    # type: (...) -> List[NotVested]
    today = dt.date.today()
    type_to_amount = {}
    for grant in parsed_history:
        amount = 0
        for vest in grant.vesting_schedule:
            if vest.date > today:
                amount += vest.amount
        if not amount:
            continue
        not_vested = type_to_amount.get(grant.type)
        if not_vested is None:
            not_vested = NotVested(type=grant.type)
            type_to_amount[grant.type] = not_vested
        not_vested.amount += amount
        money = amount * (grant.current_sell_price - grant.price_for_receiver)
        if grant.currency == finance_const.RUB:
            not_vested.in_rub += money
            not_vested.in_usd += money * to_usd
        elif grant.currency == finance_const.USD:
            not_vested.in_usd += money
            not_vested.in_rub += money * to_rub
        else:
            log.warning(
                'Unknown currency {} for grant {}'
                .format(grant.currency, grant.id)
            )
    return list(type_to_amount.values())
