# coding: utf-8
from itertools import chain
import json

from django import forms
from django.core import validators
from django.db.models import QuerySet

from review.core import const as core_const
from review.core import models as core_models
from review.gradient import models as gradient_models
from review.lib import forms as lib_forms
from review.oebs import models as oebs_models
from review.staff import const as staff_const
from review.staff import models as staff_models


class BaseParams(forms.Form):
    fields = forms.MultipleChoiceField(
        choices=core_const.FIELDS.CHOICES,
        required=False,
    )

    # вообще не нужно их получать в форме — они отфильтруются потом при
    # получении данных — сейчас лишние запросы
    VALUES_LIST_FOR_MODELS = {
        staff_models.Person: 'login',
        staff_models.Department: 'path',
        core_models.Review: 'id',
        core_models.PersonReview: 'id',
        core_models.MarksScale: 'id',
        core_models.Calibration: 'id',
        oebs_models.Currency: 'key',
        oebs_models.Profession: 'key',
        gradient_models.Umbrella: 'id',
        gradient_models.MainProduct: 'id',
    }

    def clean_fields(self):
        fields = self.cleaned_data['fields']
        for field in fields:
            if field in core_const.FIELDS.GROUPS:
                return list(core_const.FIELDS.GROUPS[field])
        return fields

    def clean(self):
        raw_cleaned_data = super(BaseParams, self).clean()
        filters = {}
        ranges = {}
        for key, value in raw_cleaned_data.items():
            if isinstance(value, QuerySet):
                values_fields = self.VALUES_LIST_FOR_MODELS[value.model]
                value = list(value.values_list(values_fields, flat=True))
            if isinstance(value, tuple(self.VALUES_LIST_FOR_MODELS)):
                id_field = self.VALUES_LIST_FOR_MODELS[value.__class__]
                value = getattr(value, id_field)
            if value in validators.EMPTY_VALUES:
                continue

            if key.endswith('_to'):
                key = key[:-len('_to')]
                range = ranges.setdefault(key, {})
                range['to'] = value
                value = range

            if key.endswith('_from'):
                key = key[:-len('_from')]
                range = ranges.setdefault(key, {})
                range['from'] = value
                value = range
            if core_const.FILTERS.PREFIX + key in core_const.FILTERS.ALL:
                key = core_const.FILTERS.PREFIX + key
            filters[key] = value
        return filters


def _get_marks_choices():
    scales = core_models.MarksScale.objects.values_list('scale', flat=True)
    scales = (json.loads(scale) for scale in scales)
    return ((it, it) for it in chain(core_const.MARK.SPECIAL_VALUES, *scales))


class PersonReviewsParams(BaseParams):
    PERSONS_MAX_COUNT = 100

    ids = forms.ModelMultipleChoiceField(
        queryset=core_models.PersonReview.objects.all(),
        required=False,
    )
    reviews = forms.ModelMultipleChoiceField(
        queryset=core_models.Review.objects.filter(
            status__in=core_const.REVIEW_STATUS.VISIBLE,
        ),
        required=False,
    )
    scale = forms.ModelMultipleChoiceField(
        queryset=core_models.MarksScale.objects.all(),
        required=False,
    )
    review_activity = lib_forms.NiceNullBooleanField(
        required=False,
    )
    persons = forms.ModelMultipleChoiceField(
        queryset=staff_models.Person.objects.all(),
        to_field_name='login',
        required=False,
    )
    calibrations = forms.ModelMultipleChoiceField(
        queryset=core_models.Calibration.objects.all(),
        required=False,
    )

    umbrella = forms.ModelMultipleChoiceField(
        queryset=gradient_models.Umbrella.objects.all(),
        required=False,
    )

    umbrella__isnull = lib_forms.NiceNullBooleanField(
        required=False,
    )

    main_product = forms.ModelMultipleChoiceField(
        queryset=gradient_models.MainProduct.objects.all(),
        required=False,
    )

    main_product__isnull = lib_forms.NiceNullBooleanField(
        required=False,
    )

    mark = forms.MultipleChoiceField(
        choices=_get_marks_choices,
        required=False,
    )
    goldstar = lib_forms.VerboseMultipleChoiceField(
        choices=core_const.GOLDSTAR.VERBOSE,
        required=False,
    )
    level_change_from = forms.IntegerField(
        min_value=core_const.VALIDATION.LEVEL_CHANGE_MIN,
        max_value=core_const.VALIDATION.LEVEL_CHANGE_MAX,
        required=False,
    )
    level_change_to = forms.IntegerField(
        min_value=core_const.VALIDATION.LEVEL_CHANGE_MIN,
        max_value=core_const.VALIDATION.LEVEL_CHANGE_MAX,
        required=False
    )
    salary_change_from = forms.DecimalField(
        min_value=core_const.VALIDATION.SALARY_CHANGE_MIN,
        max_value=core_const.VALIDATION.SALARY_CHANGE_MAX,
        required=False
    )
    salary_change_to = forms.DecimalField(
        min_value=core_const.VALIDATION.SALARY_CHANGE_MIN,
        max_value=core_const.VALIDATION.SALARY_CHANGE_MAX,
        required=False,
    )

    bonus = lib_forms.NiceNullBooleanField(
        required=False,
    )
    options_rsu = lib_forms.NiceNullBooleanField(
        required=False,
    )
    tag_average_mark = forms.ModelMultipleChoiceField(
        queryset=core_models.PersonReview.objects.order_by('tag_average_mark').distinct('tag_average_mark'),
        to_field_name='tag_average_mark',
        required=False,
    )
    taken_in_average = lib_forms.NiceNullBooleanField(
        required=False,
    )

    departments = forms.ModelMultipleChoiceField(
        queryset=staff_models.Department.objects.all(),
        to_field_name='slug',
        required=False,
    )
    profession = forms.ModelMultipleChoiceField(
        queryset=oebs_models.Profession.objects.all(),
        to_field_name='key',
        required=False,
    )
    level_changed = lib_forms.NiceNullBooleanField(
        required=False,
    )
    level_from = forms.IntegerField(
        min_value=core_const.VALIDATION.LEVEL_CHANGE_MIN,
        max_value=core_const.VALIDATION.LEVEL_CHANGE_MAX,
        required=False,
    )
    level_to = forms.IntegerField(
        min_value=core_const.VALIDATION.LEVEL_CHANGE_MIN,
        max_value=core_const.VALIDATION.LEVEL_CHANGE_MAX,
        required=False,
    )
    salary_value_from = forms.DecimalField(
        min_value=core_const.VALIDATION.SALARY_MIN,
        max_value=core_const.VALIDATION.SALARY_MAX,
        required=False,
    )
    salary_value_to = forms.DecimalField(
        min_value=core_const.VALIDATION.SALARY_MIN,
        max_value=core_const.VALIDATION.SALARY_MAX,
        required=False,
    )
    salary_currency = forms.ModelChoiceField(
        queryset=oebs_models.Currency.objects.all(),
        to_field_name='key',
        required=False,
    )
    salary_date = forms.DateField(
        required=False,
    )
    person_join_at_from = forms.DateField(
        required=False,
    )
    person_join_at_to = forms.DateField(
        required=False,
    )
    status = lib_forms.VerboseMultipleChoiceField(
        choices=core_const.PERSON_REVIEW_STATUS.VERBOSE,
        required=False,
    )
    flagged = lib_forms.NiceNullBooleanField(
        required=False,
    )
    action_at = forms.ModelMultipleChoiceField(
        queryset=staff_models.Person.objects.all(),
        to_field_name='login',
        required=False,
    )
    # is_differ_from_default = lib_forms.NiceNullBooleanField(
    #     required=False,
    # )
    subordination = lib_forms.VerboseChoiceField(
        choices=staff_const.SUBORDINATION.VERBOSE,
        required=False
    )
    subordination_subject = forms.ModelChoiceField(
        queryset=staff_models.Person.objects.all(),
        to_field_name='login',
        required=False,
    )
    reviewer = forms.ModelChoiceField(
        queryset=staff_models.Person.objects.all(),
        to_field_name='login',
        required=False,
    )

    def clean_mark(self):
        value = self.cleaned_data['mark']
        # CIA-991: Объединить оценки B и B* для фильтрации
        if value and 'B' in value:
            value += [core_const.MARK.B_STAR]
        return value

    def clean_persons(self):
        value = self.cleaned_data['persons']
        if len(value) > self.PERSONS_MAX_COUNT:
            msg = '%d or less persons allowed' % self.PERSONS_MAX_COUNT
            raise validators.ValidationError(msg)
        return value

    def clean_tag_average_mark(self):
        value = self.cleaned_data['tag_average_mark']
        if value:
            value = [it.tag_average_mark for it in value]
        return value
