# coding: utf-8
from django.db import models

from review.core import const
from review.lib.serializers import BaseSerializer, F
from review.core import serializers_extended


def serialize_params(filters):
    result = {}
    for key, value in filters.items():
        if key == const.FILTERS.REVIEW_STATUSES:
            value = [
                const.REVIEW_STATUS.VERBOSE[status]
                for status in value
            ]
        if isinstance(value, models.Model):
            value = value.id
        if isinstance(value, set):
            value = list(value)
        result[key] = value
    return result


class LogCommentSerializer(BaseSerializer):
    fields = {
        'id',
        F('text', source='text_wiki'),
        'created_at',
        'updated_at',
        F(
            'subject',
            complex=serializers_extended.ExtendedPersonSerializer,
            source='subject_*',
            complex_fields=serializers_extended.ExtendedPersonSerializer.default_fields,
        ),
        'person_review_id',
        'actions',
        F('type', source='get_type()'),
        F('subject_type', source='get_subject_type()')
    }

    @classmethod
    def get_type(cls, obj):
        return 'comment'

    @classmethod
    def get_subject_type(cls, obj):
        return const.PERSON_REVIEW_CHANGE_TYPE.VERBOSE[obj.subject_type]


class LogChangeSerializer(BaseSerializer):
    fields = {
        'id',
        F('diff', source='get_diff()'),
        'created_at',
        F(
            'subject',
            complex=serializers_extended.ExtendedPersonSerializer,
            source='subject_*',
            complex_fields=serializers_extended.ExtendedPersonSerializer.default_fields,
        ),
        'person_review_id',
        F('type', source='get_type()'),
        F('subject_type', source='get_subject_type()')
    }

    @classmethod
    def get_diff(cls, change):
        diff = change.diff
        verbosed_diff = {}
        for key, value in diff.items():
            if key == const.FIELDS.STATUS:
                value = {
                    'old': const.PERSON_REVIEW_STATUS.VERBOSE[value['old']],
                    'new': const.PERSON_REVIEW_STATUS.VERBOSE[value['new']],
                }
            elif key == const.FIELDS.GOLDSTAR:
                value = {
                    'old': const.GOLDSTAR.VERBOSE[value['old']],
                    'new': const.GOLDSTAR.VERBOSE[value['new']],
                }
            verbosed_diff[key] = value
        return verbosed_diff

    @classmethod
    def get_type(cls, obj):
        return 'change'

    @classmethod
    def get_subject_type(cls, obj):
        return const.PERSON_REVIEW_CHANGE_TYPE.VERBOSE[obj.subject_type]
