# coding: utf-8
from collections import defaultdict

from django import forms
from django.conf import settings
from django.core import validators
from django.core.exceptions import ValidationError
from django.db.models import Model
from waffle import switch_is_active

from review.core.logic import assemble
from review.core.logic import bulk
from review.core import tasks
from review.gradient import models as gradient_models
from review.lib import (
    views,
    forms as lib_forms,
)
from review.shortcuts import serializers
from review.shortcuts import const
from review.shortcuts import models

FIELDS = const.FIELDS
FILTERS = const.FILTERS


class ActionParams(forms.Form):
    approve = lib_forms.NiceNullBooleanField(
        required=False,
    )
    unapprove = lib_forms.NiceNullBooleanField(
        required=False,
    )
    allow_announce = lib_forms.NiceNullBooleanField(
        required=False,
    )
    announce = lib_forms.NiceNullBooleanField(
        required=False,
    )
    mark = forms.CharField(
        required=False,
    )
    goldstar = lib_forms.VerboseChoiceField(
        choices=const.GOLDSTAR.VERBOSE,
        required=False,
    )
    level_change = forms.IntegerField(
        min_value=const.VALIDATION.LEVEL_CHANGE_MIN,
        max_value=const.VALIDATION.LEVEL_CHANGE_MAX,
        required=False,
    )
    salary_change = forms.DecimalField(
        min_value=const.VALIDATION.SALARY_CHANGE_MIN,
        max_value=const.VALIDATION.SALARY_CHANGE_MAX,
        required=False,
    )
    salary_change_absolute = forms.IntegerField(
        min_value=const.VALIDATION.SALARY_CHANGE_ABSOLUTE_MIN,
        max_value=const.VALIDATION.SALARY_CHANGE_ABSOLUTE_MAX,
        required=False,
    )
    bonus = forms.DecimalField(
        min_value=const.VALIDATION.BONUS_MIN,
        max_value=const.VALIDATION.BONUS_MAX,
        required=False,
    )
    bonus_absolute = forms.IntegerField(
        min_value=const.VALIDATION.BONUS_ABSOLUTE_MIN,
        max_value=const.VALIDATION.BONUS_ABSOLUTE_MAX,
        required=False,
    )
    bonus_rsu = forms.IntegerField(
        min_value=const.VALIDATION.BONUS_RSU_MIN,
        max_value=const.VALIDATION.BONUS_RSU_MAX,
        required=False,
    )
    deferred_payment = forms.IntegerField(
        min_value=const.VALIDATION.DEFERRED_PAYMENT_MIN,
        max_value=const.VALIDATION.DEFERRED_PAYMENT_MAX,
        required=False,
    )
    options_rsu = forms.IntegerField(
        min_value=const.VALIDATION.OPTIONS_RSU_MIN,
        max_value=const.VALIDATION.OPTIONS_RSU_MAX,
        required=False,
    )
    flagged = lib_forms.NiceNullBooleanField(
        required=False,
    )
    flagged_positive = lib_forms.NiceNullBooleanField(
        required=False,
    )
    comment = forms.CharField(
        required=False,
    )
    reviewers = forms.Field(
        required=False
    )
    tag_average_mark = forms.CharField(
        required=False,
    )
    taken_in_average = lib_forms.NiceNullBooleanField(
        required=False,
    )
    umbrella = forms.IntegerField(
        required=False,
    )
    main_product = forms.IntegerField(
        required=False,
    )

    def clean_umbrella(self):
        if not self.cleaned_data.get('umbrella'):
            return None

        if self.cleaned_data['umbrella'] < 0:
            return -1

        return gradient_models.Umbrella.objects.get(id=self.cleaned_data['umbrella'])

    def clean_main_product(self):
        if not self.cleaned_data.get('main_product'):
            return None

        if self.cleaned_data['main_product'] < 0:
            return -1

        return gradient_models.MainProduct.objects.get(id=self.cleaned_data['main_product'])

    def clean(self):
        raw_cleaned_data = super(ActionParams, self).clean()
        cleaned_data = {
            key: value
            for key, value in raw_cleaned_data.items()
            if value not in validators.EMPTY_VALUES
        }
        if not cleaned_data:
            raise forms.ValidationError('No known action chosen')
        if 'reviewers' in cleaned_data:
            reviewers_action = cleaned_data['reviewers']
            reviewers_action_params = BulkActionEditReviewers(data=reviewers_action)
            if not reviewers_action_params.is_valid():
                self.add_error(self.reviewers, reviewers_action_params.errors)
            cleaned_data['reviewers'] = reviewers_action_params.cleaned_data
        return cleaned_data


class BulkActionParams(ActionParams):
    ids = forms.ModelMultipleChoiceField(
        queryset=models.PersonReview.objects.all(),
        required=True,
    )

    def clean_ids(self):
        models_qs = self.cleaned_data['ids']
        return models_qs.values_list('id', flat=True)

    def clean(self):
        cleaned = super(BulkActionParams, self).clean()
        return cleaned


class DetailActionParams(ActionParams):
    pass


class BulkActionEditReviewers(forms.Form):
    ids = forms.Field(required=False)
    person = forms.Field(required=False)
    person_to = forms.Field(required=False)
    position = forms.Field(required=False)

    def clean(self):
        action_type = self.data['type']
        action_map = {
            const.PERSON_REVIEW_ACTIONS.REVIEW_CHAIN_DELETE: BulkActionChainDelParams,
            const.PERSON_REVIEW_ACTIONS.REVIEW_CHAIN_REPLACE: BulkActionChainReplaceParams,
            const.PERSON_REVIEW_ACTIONS.REVIEW_CHAIN_ADD: BulkActionChainAddParams,
            const.PERSON_REVIEW_ACTIONS.REVIEW_CHAIN_REPLACE_ALL: BulkActionChainReplaceAllParams,
        }
        if action_type not in list(action_map.keys()):
            raise ValidationError("Bulk action {} not found".format(action_type))
        result = action_map[action_type](data=self.data)
        if not result.is_valid():
            self.add_error(field=None, error=result.errors)
        return result.cleaned_data


class BulkActionChainParams(forms.Form):
    person = forms.ModelChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
        required=True,
    )


class BulkActionChainDelParams(forms.Form):
    persons = forms.ModelMultipleChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
        required=False,
    )

    person = forms.ModelChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
        required=False,
    )

    def clean(self):
        data = super(BulkActionChainDelParams, self).clean()
        data['type'] = const.PERSON_REVIEW_ACTIONS.REVIEW_CHAIN_DELETE

        data['persons'] = list(data['persons']) if 'persons' in data else []

        if data.get('person', None):
            data['persons'].append(data.pop('person'))

        if not data['persons']:
            raise ValidationError('No persons to delete')

        return data


class BulkActionChainReplaceParams(BulkActionChainParams):
    person_to = forms.ModelChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
        required=True,
    )

    def clean(self):
        data = super(BulkActionChainReplaceParams, self).clean()
        data['type'] = const.PERSON_REVIEW_ACTIONS.REVIEW_CHAIN_REPLACE
        return data


class BulkActionChainAddParams(BulkActionChainParams):
    position = forms.ChoiceField(
        choices=[(x, x) for x in const.PERSON_REVIEW_ACTIONS.ADD_POSITION_CHOICE],
        required=True,
    )
    person_to = forms.ModelChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
        required=False,
    )

    def clean(self):
        data = super(BulkActionChainAddParams, self).clean()
        if data['position'] in (
                const.PERSON_REVIEW_ACTIONS.ADD_POSITION_BEFORE,
                const.PERSON_REVIEW_ACTIONS.ADD_POSITION_AFTER,
                const.PERSON_REVIEW_ACTIONS.ADD_POSITION_SAME,
        ) and data['person_to'] is None:
            self.add_error(self.person_to, "Field required")
        position_person = data.pop('person_to', None)
        if position_person is not None:
            data['position_person'] = position_person
        data['type'] = const.PERSON_REVIEW_ACTIONS.REVIEW_CHAIN_ADD
        return data


class BulkActionChainReplaceAllParams(forms.Form):
    person_to = forms.ModelChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
        required=True,
    )

    def clean(self):
        data = super(BulkActionChainReplaceAllParams, self).clean()
        data['type'] = const.PERSON_REVIEW_ACTIONS.REVIEW_CHAIN_REPLACE_ALL
        return data


# views
class ActionView(views.View):

    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {
        'id',
        'ids',
        'approve',
        'unapprove',
        'allow_announce',
        'announce',
        'reviewers',
        'tag_average_mark',
        'taken_in_average',
        'umbrella',
        'main_product',
    }

    def get_ids(self, params):
        raise NotImplementedError()

    def build_response(self, subject, bulk_result):
        updated_person_reviews = assemble.get_person_reviews(
            subject=subject,
            filters_chosen={
                FILTERS.IDS: [
                    pr.id for pr, actions_result in bulk_result.items()
                    if not actions_result['failed']
                ],
            },
            fields_requested=FIELDS.FOR_DOING_ACTION,
            role_types=(
                const.ROLE.CALIBRATION.ALL |
                const.ROLE.PERSON_REVIEW_LIST_RELATED -
                # микрооптимизация — себя никогда нельзя редактировать
                {const.ROLE.PERSON.SELF} |
                {const.ROLE.GLOBAL.ROBOT}
            ),
        )
        updated_person_reviews = {pre.id: pre for pre in updated_person_reviews}

        serialized = {}
        errs = defaultdict(lambda: dict(failed=[]))
        for person_review, actions_result in bulk_result.items():
            if person_review.id in updated_person_reviews:
                serialized_pr = serializers.PersonReviewExtendedSerializer.serialize(
                    obj=updated_person_reviews[person_review.id],
                    fields_requested=FIELDS.FOR_DOING_ACTION,
                )
            else:
                serialized_pr = {}
            serialized[person_review.id] = dict(person_review=serialized_pr)

            failed = actions_result['failed']
            for action, err_type in failed.items():
                if err_type != const.NO_ACCESS:
                    code = err_type
                else:
                    code = const.ERROR_CODES.ACTION_UNAVAILABLE
                err_obj = errs[code]
                err_obj['code'] = code
                err_obj['failed'].append((person_review.id, action))
        if errs:
            serialized['errors'] = errs
        return serialized

    def process_post(self, auth, data):
        ids = self.get_ids(data)
        bulk_result = bulk.bulk_same_action_set(
            subject=auth.user,
            ids=ids,
            params=data,
        )
        return self.build_response(auth.user, bulk_result)


class ActionBulkView(ActionView):
    form_cls_post = BulkActionParams

    def get_ids(self, params):
        return [int(id) for id in params.pop('ids')]

    def process_post(self, auth, data):
        ids = self.get_ids(data)
        need_delay = (
            switch_is_active('bulk_actions_enable_delay') and
            len(ids) > settings.BULK_ACTIONS_DELAY_THRESHOLD
        )
        raw_request_data = data.pop('raw_request_data', {})
        result = {}

        if need_delay:
            raw_request_data.pop('ids', None)
            async_result = tasks.bulk_same_action_set_task.delay(
                subject_id=auth.user.id,
                ids=ids,
                params=raw_request_data,
            )
            result['status'] = 'pending'
            result['task_id'] = async_result.id
        else:
            bulk_result = bulk.bulk_same_action_set(subject=auth.user, ids=ids, params=data)
            result['status'] = 'done'
            result.update(self.build_response(auth.user, bulk_result))

        return result

    def dispatch_process(self, request, *args, **kwargs):
        # HACK We need to pass raw data to delay task instead of model instances and so on.
        # It is an only way to get this data into a process_post method.
        kwargs.setdefault('raw_request_data', self.get_query_dict(request))
        return super(ActionBulkView, self).dispatch_process(request, *args, **kwargs)


class ActionDetailView(ActionView):
    form_cls_post = DetailActionParams

    def get_ids(self, data):
        return [int(data.pop('id'))]

    def get_logging_context(self, data):
        return {
            'instance': "PersonReview:{}".format(data['id'])
        }

    def build_response(self, subject, bulk_result):
        response = super(ActionDetailView, self).build_response(
            subject=subject,
            bulk_result=bulk_result,
        )
        errs = response.pop('errors', None)
        result = list(response.values())[0]
        if errs:
            result['errors'] = errs
        return result
