# coding: utf-8
from django import forms
from django.views.decorators import csrf
from django.utils.decorators import method_decorator

from review.core.logic import assemble
from review.frontend import forms_common
from review.lib import views
from review.lib import forms as lib_forms
from review.lib import errors
from review.shortcuts import serializers
from review.shortcuts import const
from review.shortcuts import models
from review.core.logic import (
    calibration_actions as actions,
    calibration_rights as rights,
)


FIELDS = const.FIELDS


class CalibrationPersonReviewParams(forms_common.PersonReviewsParams):
    calibration_discuss = lib_forms.NiceNullBooleanField(
        required=False
    )


class CalibrationAddPersonsForm(forms.Form):
    persons = forms.ModelMultipleChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name="login"
    )

    def clean(self):
        cleaned = super(CalibrationAddPersonsForm, self).clean()
        cleaned['persons'] = {pc.login for pc in cleaned['persons']}
        return cleaned


class CalibrationAddPersonReviewsForm(forms.Form):
    ids = forms.ModelMultipleChoiceField(
        queryset=models.PersonReview.objects.all()
    )

    def clean(self):
        cleaned = super(CalibrationAddPersonReviewsForm, self).clean()
        cleaned['ids'] = {pc.id for pc in cleaned['ids']}
        return cleaned


class CalibrationPersonReviewDeleteForm(forms.Form):
    ids = forms.ModelMultipleChoiceField(
        queryset=models.PersonReview.objects.all()
    )

    def clean(self):
        cleaned = super(CalibrationPersonReviewDeleteForm, self).clean()
        cleaned['ids'] = {pc.id for pc in cleaned['ids']}
        return cleaned


@method_decorator(csrf.csrf_exempt, name='dispatch')
class CalibrationModeCalibrationView(views.View):
    form_cls_get = form_cls_post = CalibrationPersonReviewParams

    def process_get(self, auth, data):
        calibration_id = data.pop('id')
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=calibration_id,
        )
        rights.ensure_has_action(calibration, rights.ACTIONS.LIST_PERSON_REVIEWS)
        data['calibration_id'] = calibration_id
        calibration_person_reviews = assemble.get_calibration_person_reviews(
            subject=auth.user,
            filters_chosen=data,
            requested_person_review_fields=FIELDS.DEFAULT_FOR_MODE_CALIBRATION
        )
        calibration_person_reviews = sorted(
            calibration_person_reviews,
            key=lambda it: (-it.updated_at.toordinal(), -it.id),
        )
        serializer = serializers.CalibrationPersonReviewSerializer
        serialized_cprs = serializer.serialize_many_with_person_review_fields(
            objects=calibration_person_reviews,
            person_review_fields=FIELDS.DEFAULT_FOR_MODE_CALIBRATION,
        )
        return {
            'filters': serializers.serialize_params(data),
            'calibration_person_reviews': serialized_cprs,
        }

    process_post = process_get


class CalibrationModeEditView(views.View):
    form_cls_get = CalibrationPersonReviewParams

    def process_get(self, auth, data):
        data['calibration_id'] = data.pop('id')
        calibration_person_reviews = assemble.get_calibration_person_reviews(
            subject=auth.user,
            filters_chosen=data,
            requested_person_review_fields=FIELDS.DEFAULT_FOR_MODE_CALIBRATION_EDIT
        )
        serializer = serializers.CalibrationPersonReviewSerializer
        serialized_crps = serializer.serialize_many_with_person_review_fields(
            objects=calibration_person_reviews,
            person_review_fields=FIELDS.DEFAULT_FOR_MODE_CALIBRATION_EDIT,
        )
        return {
            'filters': serializers.serialize_params(data),
            'calibration_person_reviews': serialized_crps,
        }


class CalibrationAddPersonReviewsView(views.View):
    form_cls_post = CalibrationAddPersonReviewsForm
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'id', 'ids'}

    def get_logging_context(self, data):
        return {
            'instance': "Calibration:{}".format(data['id'])
        }

    @staticmethod
    def get_person_reviews_ids(auth, data):
        return assemble.get_person_review_ids(
            subject=auth.user,
            filters={
                const.FILTERS.IDS: data['ids'],
                const.FILTERS.REVIEW_ACTIVITY: True,
            },
        )

    @staticmethod
    def get_non_available(data, person_reviews):
        available = {pr['id'] for pr in person_reviews}
        return 'ids', set(data['ids']) - available

    def process_post(self, auth, data):
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=data['id'],
        )
        rights.ensure_has_action(calibration, rights.ACTIONS.EDIT_CALIBRATORS)
        person_reviews_ids = self.get_person_reviews_ids(auth, data)
        created, existed, with_roles, person_reviews = actions.add_person_reviews(
            subject=auth.user,
            calibration=calibration,
            person_reviews_ids=person_reviews_ids,
        )
        non_available_field, non_available_values = self.get_non_available(data, person_reviews)
        res = {'success': list(created)}

        codes = const.ERROR_CODES
        if existed:
            res['warnings'] = {
                codes.ALREADY_EXISTS: {
                    'code': codes.ALREADY_EXISTS,
                    'logins': existed,
                }
            }

        has_errs = non_available_values or with_roles
        if has_errs:
            res['errors'] = {}
            if non_available_values:
                res['errors'][codes.PERMISSION_DENIED] = {
                    'code': codes.PERMISSION_DENIED,
                    non_available_field: list(non_available_values),
                }
            if with_roles:
                role_to_err_type = {
                    const.ROLE.CALIBRATION.CALIBRATOR: codes.PERSONS_ARE_CALIBRATORS,
                    const.ROLE.CALIBRATION.ADMIN: codes.PERSONS_ARE_CALIBRATION_ADMINS,
                }
                for role, logins in with_roles.items():
                    code = role_to_err_type[role]
                    res['errors'][code] = {'code': code, 'logins': logins}

        return res


class CalibrationAddPersonsView(CalibrationAddPersonReviewsView):
    form_cls_post = CalibrationAddPersonsForm
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'id', 'persons'}

    @staticmethod
    def get_person_reviews_ids(auth, data):
        return assemble.get_person_review_ids(
            subject=auth.user,
            filters={
                const.FILTERS.REVIEW_ACTIVITY: True,
                const.FILTERS.PERSONS: data['persons'],
            }
        )

    @staticmethod
    def get_non_available(data, person_reviews):
        available = {pr['person__login'] for pr in person_reviews}
        return 'logins', set(data['persons']) - available


class CalibrationDeletePersonReviewView(views.View):
    form_cls_post = CalibrationPersonReviewDeleteForm
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'id', 'ids'}

    def get_logging_context(self, data):
        return {
            'instance': "Calibration:{}".format(data['id'])
        }

    def process_post(self, auth, data):
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=data['id'],
        )
        rights.ensure_has_action(calibration, rights.ACTIONS.EDIT_CALIBRATORS)
        calibration_person_reviews = assemble.get_calibration_person_reviews(
            subject=auth.user,
            filters_chosen={
                'calibration_id': data['id'],
                'person_review_ids': data['ids']
            },
            requested_person_review_fields={FIELDS.ID}
        )
        if not calibration_person_reviews:
            raise errors.PermissionDenied(
                type='calibration_person_reviews',
                perm='delete',
            )
        ids_to_delete = {pc.id for pc in calibration_person_reviews}
        models.CalibrationPersonReview.objects.filter(
            id__in=ids_to_delete
        ).delete()
        return {}
