# coding: utf-8
from django import forms

from review.core import (
    const,
    models as core_models,
)
from review.core.logic import (
    assemble,
    bulk,
)
from review.lib import forms as lib_forms
from review.lib import views
from review.shortcuts import serializers


class CalibrationPersonReviewActionParamsForm(forms.Form):
    discuss = lib_forms.NiceNullBooleanField(required=False)


class CalibrationPersonReviewBulkActionParamsForm(CalibrationPersonReviewActionParamsForm):
    ids = forms.ModelMultipleChoiceField(
        queryset=core_models.CalibrationPersonReview.objects.all()
    )

    def clean(self):
        cleaned = super(CalibrationPersonReviewBulkActionParamsForm, self).clean()
        queried_models = cleaned.pop('ids')
        cleaned['ids'] = set(queried_models.values_list('id', flat=True))
        return cleaned


class CalibrationPersonReviewActionView(views.View):
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {
        'id',
        'ids',
        'discuss',
    }

    def __init__(self, *args, **kwargs):
        super(CalibrationPersonReviewActionView, self).__init__(*args, **kwargs)
        self._ids = None

    def set_ids(self, params):
        raise NotImplementedError()

    def process_post(self, auth, data):
        self.set_ids(data)
        diffs = bulk.bulk_same_action_set_for_calibration_person_review(
            subject=auth.user,
            ids=self._ids,
            actions=data,
        )

        failed = []
        for id, diff in diffs.items():
            for action in diff['failed']:
                failed.append((id, action))

        calibration_person_reviews = assemble.get_calibration_person_reviews(
            subject=auth.user,
            filters_chosen=dict(ids=self._ids),
            requested_person_review_fields=const.FIELDS.DEFAULT_FOR_MODE_CALIBRATION
        )
        serializer = serializers.CalibrationPersonReviewSerializer
        serialized_cprs = serializer.serialize_many_with_person_review_fields(
            objects=sorted(calibration_person_reviews, key=lambda it: it.id),
            person_review_fields=const.FIELDS.DEFAULT_FOR_MODE_CALIBRATION,
        )
        response = dict(calibration_person_reviews=serialized_cprs)
        if failed:
            response['errors'] = {
                const.ERROR_CODES.ACTION_UNAVAILABLE: dict(
                    code=const.ERROR_CODES.ACTION_UNAVAILABLE,
                    failed=failed,
                )
            }
        return response


class CalibrationPersonReviewBulkActionView(CalibrationPersonReviewActionView):
    form_cls_post = CalibrationPersonReviewBulkActionParamsForm

    def set_ids(self, params):
        if self._ids is None:
            self._ids = params.pop('ids')
        return self._ids


class CalibrationPersonReviewDetailActionView(CalibrationPersonReviewActionView):
    form_cls_post = CalibrationPersonReviewActionParamsForm

    def set_ids(self, params):
        if self._ids is None:
            self._ids = [int(params.pop('id'))]
        return self._ids

    def process_post(self, auth, data):
        res = super(CalibrationPersonReviewDetailActionView, self).process_post(auth, data)
        cprs = res.pop('calibration_person_reviews')
        res['calibration_person_review'] = cprs[0] if cprs else {}
        return res

    def get_logging_context(self, data):
        return {
            'instance': "Calibration:{}".format(data['id'])
        }
