# coding: utf-8
from typing import Dict, List

from django import forms

from review.core.logic import assemble
from review.core.logic import calibration_actions
from review.core.logic import roles
from review.core.logic import domain_objs
from review.lib import views
from review.lib import forms as lib_forms
from review.lib import errors
from review.shortcuts import serializers
from review.shortcuts import const
from review.shortcuts import models


FIELDS = const.FIELDS
FILTERS = const.FILTERS


class CalibrationCreateForm(forms.Form):
    name = forms.CharField(max_length=255)
    start_date = forms.DateField()
    finish_date = forms.DateField()
    person_reviews = forms.ModelMultipleChoiceField(
        queryset=models.PersonReview.objects.all()
    )
    admins = forms.ModelMultipleChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
        required=False,
    )


class CalibrationListForm(forms.Form):
    statuses = lib_forms.VerboseMultipleChoiceField(
        choices=const.CALIBRATION_STATUS.CHOICES,
        required=False
    )
    fields = lib_forms.VerboseMultipleChoiceField(
        choices=[['author', 'author']],
        required=False
    )


class CalibrationList(views.View):
    form_cls_post = CalibrationCreateForm
    form_cls_get = CalibrationListForm
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {
        'name',
        'start_date',
        'finish_date',
        'person_reviews',
        'admins',
    }

    def process_post(self, auth, data):
        if not roles.has_global_roles(
            person=auth.user,
            global_roles=[const.ROLE.GLOBAL.CALIBRATION_CREATOR],
        ):
            raise errors.PermissionDenied(
                action='create',
                type='calibration',
            )
        person_review_ids = {pr.id for pr in data['person_reviews']}
        person_reviews = assemble.get_person_reviews(
            subject=auth.user,
            filters_chosen={FILTERS.IDS: person_review_ids},
            fields_requested={FIELDS.ID},
        )
        if not person_reviews:
            raise errors.PermissionDenied(
                action='create',
                type='calibration',
            )
        calibration = calibration_actions.create(
            person=auth.user,
            person_reviews=person_reviews,
            name=data['name'],
            start_date=data['start_date'],
            finish_date=data['finish_date'],
            admins=data['admins'],
        )
        return {'id': calibration.id}

    def process_get(self, auth, data):
        required_fields = set(data.get('fields', ())) | FIELDS.DB_CALIBRATION_FIELDS
        calibration_list = assemble.get_calibrations(
            subject=auth.user,
            filters=data,
            requested_fields=required_fields,
        )
        calibration_list = sorted(
            calibration_list,
            key=lambda it: (
                const.CALIBRATION_STATUS.ORDERING.index(it.status),
                -it.start_date.toordinal()
            ),
        )
        return {
            "filters": serializers.serialize_params(data),
            "calibrations": serializers.ExtendedCalibrationSerializer.serialize_many(
                calibration_list,
                context=dict(language=auth.user.language),
            ),
        }


class CalibrationSuggestForm(forms.Form):
    text = forms.CharField(required=False)
    reviews = forms.ModelMultipleChoiceField(
        queryset=models.Review.objects.filter(
            status__in=const.REVIEW_STATUS.VISIBLE,
        ),
        required=False,
    )
    review_activity = lib_forms.NiceNullBooleanField(
        required=False,
    )
    role = lib_forms.VerboseChoiceField(
        choices=const.ROLE.CALIBRATION.CHOICES,
        required=False,
    )


class CalibrationSuggestView(views.View):
    form_cls_get = CalibrationSuggestForm

    def process_get(self, auth, data):
        role_types = [data['role']] if data.get('role') else None
        calibration_list = assemble.get_calibrations(
            subject=auth.user,
            filters={},
            role_types=role_types,
            requested_fields=FIELDS.DB_CALIBRATION_FIELDS,
        )

        calibration_id_to_reviews = assemble.get_calibrations_review_info(
            calibration_list,
            filters=data,
        )

        res = []
        search_words = data['text'].lower().split(' ')
        for calibration in calibration_list:
            reviews = calibration_id_to_reviews.get(calibration.id) or []
            accepted = any(
                self.is_string_suitable(
                    (review['name'] + calibration.name).lower(),
                    search_words,
                )
                for review in reviews
            )

            if accepted:
                res.append(self.serialize(
                    calibration,
                    reviews,
                    auth.user.language,
                ))

        return {'calibrations': res}

    @staticmethod
    def is_string_suitable(check_string, search_words):
        res = all(
            word in check_string
            for word in search_words
        )
        return res

    @staticmethod
    def serialize(calibration, reviews, lang):
        # type: (domain_objs.Calibration, List[Dict], str) -> Dict
        res = serializers.ExtendedCalibrationSerializer.serialize(
            calibration,
            context=dict(language=lang),
        )
        res['reviews'] = reviews
        return res
