# coding: utf-8
from django import forms
from django import http

from review.lib import forms as lib_forms
from review.lib import views
from review.core.logic import (
    assemble,
    calibration_actions as actions,
    calibration_rights as rights,
)
from review.shortcuts import const
from review.shortcuts import models
from review.shortcuts import serializers


FIELDS = const.FIELDS
FILTERS = const.FILTERS


class CalibrationEditForm(forms.Form):
    name = forms.CharField(required=False)
    start_date = forms.DateField(required=False)
    finish_date = forms.DateField(required=False)
    admins = forms.ModelMultipleChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
        required=False,
    )


class CalibrationWorkflowForm(forms.Form):
    workflow = lib_forms.VerboseChoiceField(
        choices=[(x, x) for x in rights.ACTIONS.STATUS]
    )


class CalibrationDetailView(views.View):
    form_cls_post = form_cls_delete = CalibrationEditForm
    log_params_for = {'post', 'delete'}
    WHITE_LIST_TO_LOG = {
        'id',
        'name',
        'start_date',
        'finish_date',
        'admins',
    }

    def get_logging_context(self, data):
        return {
            'instance': "Calibration:{}".format(data['id'])
        }

    def process_get(self, auth, data):
        calibration_id = data['id']
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=calibration_id,
            requested_fields={'author'} | FIELDS.DB_CALIBRATION_FIELDS,
        )
        if calibration is None:
            raise http.Http404('id %s not found' % calibration_id)
        calibration.admins = actions.get_admins(calibration=calibration)
        calibration.review_ids = actions.get_review_ids(calibration=calibration)
        return serializers.ExtendedCalibrationSerializer.serialize(
            obj=calibration,
            context=dict(language=auth.user.language),
        )

    def process_post(self, auth, data):
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=data['id'],
        )
        rights.ensure_has_action(calibration, rights.ACTIONS.EDIT_PARAMETERS)
        calibration = actions.edit_parameters(calibration, data)
        return {'id': calibration.id}

    def process_delete(self, auth, data):
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=data['id'],
        )
        rights.ensure_has_action(calibration, rights.ACTIONS.DELETE)
        models.Calibration.objects.filter(id=data['id']).delete()
        return {'id': calibration.id}


class CalibrationCopyView(views.View):
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'id'}

    def get_logging_context(self, data):
        return {
            'instance': "Calibration:{}".format(data['id'])
        }

    def process_post(self, auth, data):
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=data['id'],
        )
        rights.ensure_has_action(calibration, rights.ACTIONS.COPY_CALIBRATION)
        return {'id': actions.copy(auth.user, calibration)}


class CalibrationFollowWorkflowView(views.View):
    form_cls_post = CalibrationWorkflowForm
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'id', 'workflow'}

    def get_logging_context(self, data):
        return {
            'instance': "Calibration:{}".format(data['id'])
        }

    def process_post(self, auth, data):
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=data['id'],
        )
        action = data['workflow']
        rights.ensure_has_action(calibration, action)
        actions.follow_workflow(calibration, action)
        return {"id": calibration.id}
