# coding: utf-8
from django import forms

from review.core.logic import (
    assemble,
    calibration_actions as actions,
    calibration_rights as rights,
)
from review.frontend.views.person_reviews import HistoryView
from review.lib import (
    errors,
    views,
)
from review.shortcuts import const
from review.shortcuts import serializers
from review.shortcuts import models


class CalibrationRolesDeleteForm(forms.Form):
    persons = forms.ModelMultipleChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
    )


class AddCalibrationAdminsForm(forms.Form):
    persons = forms.ModelMultipleChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
    )


class AddCalibrationCalibratorsForm(forms.Form):
    persons = forms.ModelMultipleChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
    )


class CalibrationCalibratorsView(views.View):

    def process_get(self, auth, data):
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=data['id'],
        )
        rights.ensure_has_action(calibration, rights.ACTIONS.LIST_CALIBRATORS)
        calibration_roles = actions.get_calibrators(calibration)
        actions.prepopulate_calibrators_data([c['person'] for c in calibration_roles], auth.user)
        return {
            'id': data['id'],
            'roles': serializers.CalibrationRolePersonSerializer.serialize_many(
                calibration_roles,
                context=dict(language=auth.user.language),
            )
        }


class CalibrationAddCalibratorsView(views.View):
    form_cls_post = AddCalibrationCalibratorsForm
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'id', 'persons'}

    def get_logging_context(self, data):
        return {
            'instance': "Calibration:{}".format(data['id'])
        }

    def process_post(self, auth, data):
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=data['id'],
        )
        rights.ensure_has_action(calibration, rights.ACTIONS.EDIT_CALIBRATORS)
        creation_result = actions.add_calibrators(
            calibration=calibration,
            persons=data['persons'],
        )
        existing_calibrators = creation_result['existing_calibrators']
        persons_in_calibration = creation_result['persons_in_calibration']
        response = dict(id=calibration.id)
        if existing_calibrators or persons_in_calibration:
            errs = response['errors'] = {}
            if existing_calibrators:
                errs[const.ERROR_CODES.ALREADY_EXISTS] = dict(
                    params=dict(logins=existing_calibrators),
                    code=const.ERROR_CODES.ALREADY_EXISTS,
                )
            if persons_in_calibration:
                errs[const.ERROR_CODES.PERSONS_ARE_CALIBRATING] = dict(
                    params=dict(logins=persons_in_calibration),
                    code=const.ERROR_CODES.PERSONS_ARE_CALIBRATING,
                )
        return response


class CalibrationDeleteCalibratorsView(views.View):
    form_cls_post = CalibrationRolesDeleteForm
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'id', 'persons'}

    def get_logging_context(self, data):
        return {
            'instance': "Calibration:{}".format(data['id'])
        }

    def process_post(self, auth, data):
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=data['id'],
        )
        rights.ensure_has_action(calibration, rights.ACTIONS.EDIT_CALIBRATORS)
        actions.delete_roles(
            calibration=calibration,
            persons=data['persons'],
            role_type=const.ROLE.CALIBRATION.CALIBRATOR,
        )
        return {'id': calibration.id}


class CalibrationAdminsView(views.View):

    def process_get(self, auth, data):
        id_ = data['id']
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=id_,
        )
        rights.ensure_has_action(calibration, rights.ACTIONS.EDIT_PARAMETERS)
        admins = actions.get_admins(calibration)
        return dict(
            id=calibration.id,
            admins=serializers.PersonSerializer.serialize_many(
                admins,
                context=dict(language=auth.user.language),
            ),
        )


class CalibrationDeleteAdminsView(views.View):
    form_cls_post = CalibrationRolesDeleteForm
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'id', 'persons'}

    def get_logging_context(self, data):
        return {
            'instance': "Calibration:{}".format(data['id'])
        }

    def process_post(self, auth, data):
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=data['id'],
        )
        rights.ensure_has_action(calibration, rights.ACTIONS.EDIT_PARAMETERS)
        actions.delete_roles(
            calibration=calibration,
            persons=data['persons'],
            role_type=const.ROLE.CALIBRATION.ADMIN,
        )
        return {'id': calibration.id}


class CalibrationAddAdminsView(views.View):
    form_cls_post = AddCalibrationAdminsForm
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'id', 'persons'}

    def get_logging_context(self, data):
        return {
            'instance': "Calibration:{}".format(data['id'])
        }

    def process_post(self, auth, data):
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=data['id'],
        )
        if not calibration:
            raise errors.PermissionDenied(
                action='read',
                type='calibration',
                id=data['id'],
            )
        rights.ensure_has_action(calibration, rights.ACTIONS.EDIT_PARAMETERS)
        actions.add_admins(
            calibration=calibration,
            persons=data['persons'],
        )
        return {'id': calibration.id}


class CalibrationRecommendedCalibratorsView(views.View):
    def process_get(self, auth, data):
        calibration = assemble.get_calibration(
            subject=auth.user,
            id=data['id'],
        )
        rights.ensure_has_action(calibration, rights.ACTIONS.EDIT_CALIBRATORS)
        calibration_roles = actions.get_recommended_calibrators(calibration)
        actions.prepopulate_calibrators_data([c['person'] for c in calibration_roles], auth.user)
        return {
            'id': data['id'],
            'persons': serializers.CalibrationRolePersonSerializer.serialize_many(
                calibration_roles,
                context=dict(language=auth.user.language),
            )
        }


class CalibrationPersonReviewsHistoryView(HistoryView):
    roles = {
        const.ROLE.PERSON_REVIEW.REVIEWER,
        const.ROLE.PERSON_REVIEW.TOP_REVIEWER,
        const.ROLE.CALIBRATION.ADMIN,
        const.ROLE.CALIBRATION.CALIBRATOR,
        const.ROLE.PERSON_REVIEW.CALIBRATOR_INHERITED,
        const.ROLE.PERSON_REVIEW.CALIBRATION_ADMIN_INHERITED,
        const.ROLE.PERSON_REVIEW.CALIBRATOR_ARCHIVED,
        const.ROLE.REVIEW.ACCOMPANYING_HR,
    }
