# coding: utf-8

from django import forms
from django import http
from django.utils import timezone
import logging
import waffle

from review.lib import views
from review.shortcuts import const
from review.shortcuts import serializers
from review.shortcuts import models
from review.core.logic import assemble
from review.xiva.update_calibration import post_to_xiva


log = logging.getLogger(__name__)


Serializer = serializers.LogCommentSerializer


class PersonReviewCommentsView(views.View):

    def process_get(self, auth, data):
        person_review = assemble.get_person_review(
            subject=auth.user,
            fields_requested={
                const.FIELDS.COMMENTS,
            },
            id=int(data['id']),
            role_types=const.ROLE.PERSON_REVIEW_LIST_RELATED | const.ROLE.CALIBRATION.ALL,
        )
        if person_review:
            serialized = Serializer.serialize_many(person_review.comments)
        else:
            serialized = []
        if len(serialized) == 0:
            log.warning('PersonReviewComments: person_review_id: %s has no comments', data['id'])
        return {
            'comments': serialized,
        }


class PersonReviewCommentsDetailView(views.View):

    class Form(forms.Form):
        text = forms.CharField(required=False)

    form_cls_post = Form
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'id', 'comment_id'}

    def process_post(self, auth, data):
        comment_id = int(data['comment_id'])
        comment = assemble.get_person_review_comment(
            subject=auth.user,
            id=comment_id,
            role_types=const.ROLE.PERSON_REVIEW_LIST_RELATED | const.ROLE.CALIBRATION_PERSON_REVIEWS_ROLES_DEFAULT
        )

        if comment is None:
            msg = 'Comment not found or not available'
            return http.HttpResponseNotFound(msg)

        if not comment.actions[const.PERSON_REVIEW_COMMENT_ACTIONS.EDIT]:
            return http.HttpResponseForbidden('Only author can edit comment')

        new_text_wiki = data['text']
        new_updated_at = timezone.now()
        # вынести в отдельный модуль, когда будем делать вики-разметку
        models.PersonReviewComment.objects.filter(
            id=comment_id,
        ).update(
            text_wiki=new_text_wiki,
            updated_at=new_updated_at,
        )
        comment.text_wiki = new_text_wiki
        comment.updated_at = new_updated_at

        if waffle.switch_is_active("calibration_auto_update"):
            post_to_xiva(auth.user.login, [data['id']])

        return {
            'comment': Serializer.serialize(comment)
        }
