# coding: utf-8
import logging
import collections

from django.http.response import HttpResponseNotFound
from django.views.decorators import csrf
from django.utils.decorators import method_decorator

from review.frontend import forms_common
from review.shortcuts import serializers
from review.shortcuts import const
from review.lib import datetimes
from review.lib import views
from review.lib import errors
from review.core.logic import assemble
from review.core.tasks import export_person_review_template_task

logger = logging.getLogger(__name__)


FIELDS = const.FIELDS


@method_decorator(csrf.csrf_exempt, name='dispatch')
class _FileExportView(views.View):
    form_cls_get = form_cls_post = forms_common.PersonReviewsParams
    FIELDS_REQUIRED_FOR_EXPORT_DATA = frozenset()
    serializer = None

    def process_get(self, auth, data):
        person_reviews = assemble.get_person_reviews(
            subject=auth.user,
            fields_requested=self.FIELDS_REQUIRED_FOR_EXPORT_DATA,
            filters_chosen=data,
            role_types=const.ROLE.PERSON_REVIEW_LIST_RELATED | const.ROLE.CALIBRATION_PERSON_REVIEWS_ROLES_DEFAULT,
        )

        if not person_reviews:
            raise errors.PermissionDenied(
                type='person_reviews',
            )

        res = self.serializer.get_many_ordered(person_reviews)
        return res

    def process_post(self, *args, **kwargs):
        return self.process_get(*args, **kwargs)


class PersonReviewExportView(_FileExportView):

    FIELDS_REQUIRED_FOR_EXPORT_DATA = frozenset([
        FIELDS.ID,
        FIELDS.PERSON_FIRST_NAME,
        FIELDS.PERSON_LAST_NAME,
        FIELDS.PERSON_LOGIN,
        FIELDS.PERSON_CITY_NAME,
        FIELDS.PERSON_DEPARTMENT_CHAIN_NAMES,

        FIELDS.REVIEW_TYPE,
        FIELDS.REVIEW_MARKS_SCALE,

        FIELDS.STATUS,
        FIELDS.ACTION_AT,
        FIELDS.FLAGGED,
        FIELDS.TAG_AVERAGE_MARK,
        FIELDS.TAKEN_IN_AVERAGE,
        FIELDS.MARK,
        FIELDS.GOLDSTAR,
        FIELDS.SALARY_CHANGE,
        FIELDS.SALARY_CHANGE_ABSOLUTE,
        FIELDS.SALARY_AFTER_REVIEW,
        FIELDS.BONUS,
        FIELDS.BONUS_ABSOLUTE,
        FIELDS.BONUS_RSU,
        FIELDS.DEFERRED_PAYMENT,
        FIELDS.OPTIONS_RSU,
        FIELDS.LEVEL_CHANGE,
        FIELDS.SHORT_HISTORY,

        FIELDS.FTE,
        FIELDS.PROFESSION,
        FIELDS.LEVEL,
        FIELDS.SALARY_CURRENCY,
        FIELDS.SALARY_VALUE,

        FIELDS.UMBRELLA,
        FIELDS.MAIN_PRODUCT,
        FIELDS.UPDATED_AT,
    ])
    serializer = serializers.PersonReviewsSerializer

    def get_file_properties(self, response_format):
        fp = super(PersonReviewExportView, self).get_file_properties(response_format)
        fp.file_name = 'reviews {:%Y-%m-%d %H:%M}'.format(datetimes.now())
        fp.sheet_name = 'Выгрузка из ревьюшницы'
        return fp


class PersonReviewTemplateExportView(PersonReviewExportView):

    def process_get(self, auth, data):
        review_id = data.get('id')

        template_type = const.EXCEL_TEMPLATE_TYPE.REVERSE_VERBOSE[data.get('template_type')]
        template = assemble.get_review_template_file(auth.user, review_id, template_type)

        if not template:
            return HttpResponseNotFound()

        export_person_review_template_task.delay(auth.user.login, data, list(self.FIELDS_REQUIRED_FOR_EXPORT_DATA))

        return {'status': 'OK'}


class CommentsExportView(_FileExportView):
    response_format = 'xls'
    serializer = serializers.CommentsSerializer

    FIELDS_REQUIRED_FOR_EXPORT_DATA = frozenset([
        FIELDS.ID,
        FIELDS.PERSON_LOGIN,
        FIELDS.COMMENTS,
    ])

    def get_file_properties(self, response_format):
        fp = super(CommentsExportView, self).get_file_properties(response_format)
        fp.file_name = 'review_comments'
        fp.sheet_name = 'comments'
        return fp
