from django import forms
from django.shortcuts import get_object_or_404
from django.utils import timezone

from review.gradient import models as gradient_models
from review.gradient import serializers
from review.lib import views
from review.shortcuts import models


class SuggestForm(forms.Form):
    text = forms.CharField()


class UmbrellaByIdsForm(forms.Form):
    ids = forms.ModelMultipleChoiceField(
        queryset=gradient_models.Umbrella.objects.all()
    )


class MainProductByIdsForm(forms.Form):
    ids = forms.ModelMultipleChoiceField(
        queryset=gradient_models.MainProduct.objects.all()
    )


class PersonsUmbrellaForm(forms.Form):
    engaged_datetime = forms.DateTimeField(required=False)


class _SuggestView(views.View):
    form_cls_get = SuggestForm
    result_field_name = None
    serializer = None

    def serialize(self, obj):
        raise NotImplementedError

    def get_base_query(self):
        raise NotImplementedError

    def get_full_name(self, obj):
        raise NotImplementedError

    def process_get(self, auth, data):
        search_words = data['text'].lower().split(' ')
        query = self.get_base_query()

        to_serialize = (
            obj for obj in query
            if self.is_string_suitable(self.get_full_name(obj), search_words)
        )
        return {
            self.result_field_name: self.serializer.serialize_many(to_serialize)
        }

    @staticmethod
    def is_string_suitable(check_string, search_words):
        res = all(
            word in check_string
            for word in search_words
        )
        return res


class UmbrellasSuggestView(_SuggestView):
    result_field_name = 'umbrellas'
    serializer = serializers.UmbrellaSerializer

    def get_full_name(self, obj):
        values = [obj.name, obj.issue_key, obj.main_product.name if obj.main_product else '']
        return '_'.join(v.lower() for v in values if v)

    def get_base_query(self):
        return gradient_models.Umbrella.objects.select_related('main_product')


class UmbrellasByIdsView(views.View):
    form_cls_get = UmbrellaByIdsForm

    def process_get(self, auth, data):
        query = (
            gradient_models.Umbrella.objects
            .select_related('main_product')
            .filter(id__in=data['ids'])
        )

        return {
            'umbrellas': serializers.UmbrellaSerializer.serialize_many(query)
        }


class MainProductSuggestView(_SuggestView):
    result_field_name = 'main_products'
    serializer = serializers.MainProductSerializer

    def get_full_name(self, obj):
        return obj.name.lower()

    def get_base_query(self):
        return gradient_models.MainProduct.objects.all()


class MainProductByIdsView(views.View):
    form_cls_get = MainProductByIdsForm

    def process_get(self, auth, data):
        query = (
            gradient_models.MainProduct.objects
            .values('id', 'name')
            .filter(id__in=data['ids'])
        )

        return {
            'main_products': list(query)
        }


class PersonUmbrellasView(views.View):
    form_cls_get = PersonsUmbrellaForm

    def process_get(self, auth, data):
        person = get_object_or_404(models.Person, login=data['login'])
        engaged_datetime = data.get('engaged_datetime') or timezone.now()

        query = (
            gradient_models.UmbrellaPerson.objects
            .filter(person=person)
            .filter_active(engaged_datetime)
            .select_related('umbrella', 'umbrella__main_product')
        )

        return {
            'person_umbrellas': serializers.UmbrellaPersonSerializer.serialize_many(query)
        }
