# coding: utf-8
from __future__ import unicode_literals

import uuid

import yenv
from django import forms
from django.conf import settings
from django.core.exceptions import ValidationError
from django.db import transaction
from django.http import HttpResponse
from django.template import Template, RequestContext

from review.core import const
from review.core.models import (
    PersonReview,
)
from review.gradient.models import (
    MainProduct,
    Umbrella,
    MainProductReview,
    UmbrellaReview,
    UmbrellaPersonReview,
)

from review.lib import (
    views,
    serializers as lib_serializers,
)


LOGINS_WHITELIST = settings.IMPORT_UMBRELLAS_LOGINS_WHITELIST


def check_access(login):
    if yenv.type in ('development', 'testing'):
        return True

    return login in LOGINS_WHITELIST


def gen_fake_issue_key():
    fake_issue_key = 'FAKE_ST_KEY__{}'.format(uuid.uuid4().hex)
    assert len(fake_issue_key) < 255

    return fake_issue_key


class VSandUmbrellasValidateForm(forms.Form):
    def __init__(self, data, review, **kwargs):
        self.review = review

        FIELDS_MAP = {
            'Логин': 'login',
            'VS': 'main_product_name',
            'Зонтик': 'umbrella_name',
            'Ревью': 'review_id'
        }

        for field_from, field_to in FIELDS_MAP.items():
            if field_from in data:
                data[field_to] = data[field_from]
                del data[field_from]

        super(VSandUmbrellasValidateForm, self).__init__(data=data, **kwargs)

    login = forms.CharField()
    main_product_name = forms.CharField()
    umbrella_name = forms.CharField()
    review_id = forms.IntegerField(min_value=1, required=False)

    def clean(self):
        cleaned = super(VSandUmbrellasValidateForm, self).clean()
        return cleaned


class UmbrellasFileForm(forms.Form):
    data_file = forms.FileField()

    def clean(self):
        if not self.is_valid():
            raise ValidationError(self.errors)
        cleaned = super(UmbrellasFileForm, self).clean()
        serialized_file = lib_serializers.serialize_file_sheet(
            cleaned['data_file'],
            VSandUmbrellasValidateForm,
        )

        return {
            'persons_list': serialized_file,
        }


class ImportUmbrellasUploadView(views.View):
    form_cls_post = UmbrellasFileForm

    include_auth_to_response = False

    json_response_params = {
        'json_dumps_params': {
            'indent': 4
        }
    }

    @transaction.atomic
    def process_post(self, auth, data):
        if not check_access(auth.user.login):
            return self.do_response(
                request=self.request,
                response={'errors': ['ACCESS DENIED']},
                status_code=400,
            )

        total_created = 0
        multiples_deleted = 0
        data_errors = []

        for person in data['persons_list']:
            login = person['login']
            main_product_name = person['main_product_name']
            umbrella_name = person['umbrella_name']
            review_id = person.get('review_id', None)

            qs = PersonReview.objects.filter(
                person__login=login,
                review__status__in=const.REVIEW_STATUS.ACTIVE,
            )
            if review_id:
                qs = qs.filter(
                    review_id=review_id,
                )
            qs_count = qs.count()

            if qs_count == 0:
                data_errors.append('ERROR: no active reviews for {}'.format(login))
                continue
            elif qs_count > 1:
                review_ids = qs.values_list('review_id', flat=True)
                data_errors.append('ERROR: {} active reviews for {}: {}'.format(
                    qs_count,
                    login,
                    ', '.join(map(str, review_ids))
                ))
                continue

            person_review = qs.first()

            main_product, _ = MainProduct.objects.get_or_create(
                name=main_product_name,
                defaults={
                    'issue_key': gen_fake_issue_key(),
                }
            )

            umbrella, _ = Umbrella.objects.get_or_create(
                name=umbrella_name,
                main_product=main_product,
                defaults={
                    'issue_key': gen_fake_issue_key(),
                }
            )

            _, _ = MainProductReview.objects.get_or_create(
                main_product=main_product,
                review=person_review.review
            )

            _, _ = UmbrellaReview.objects.get_or_create(
                umbrella=umbrella,
                review=person_review.review
            )

            umbrella_person_review, umb_pr_created = UmbrellaPersonReview.objects.get_or_create(
                umbrella=umbrella,
                person_review=person_review
            )

            umb_pr_deleted, _ = UmbrellaPersonReview.objects.filter(
                person_review=person_review,
            ).exclude(
                umbrella=umbrella
            ).delete()

            multiples_deleted += umb_pr_deleted

            if umb_pr_created:
                total_created += 1

        if data_errors:
            transaction.set_rollback(True)

            return self.do_response(
                request=self.request,
                response={'errors': data_errors},
                status_code=400,
            )

        return {
            'message': 'Total {} created, {} deleted.'.format(
                total_created,
                multiples_deleted,
            ),
        }


def import_umbrellas(request):
    if not check_access(request.auth.user.login):
        return HttpResponse('Access denied', status=400)

    tmpl = Template('''
    <form action='upload/' method='POST' enctype='multipart/form-data'>
    {% csrf_token %}
    {{ form }}
    <input type='submit' value='⛱ Загрузить ⛱'>
    </form>
    ''')
    context = RequestContext(request, {'form': UmbrellasFileForm()})
    return HttpResponse(tmpl.render(context))
