# coding: utf-8

import logging
import operator

from review.lib import views
from review.lib.serializers import F
from review.shortcuts import const
from review.shortcuts import serializers
from review.core.logic import assemble

FIELDS = const.FIELDS


log = logging.getLogger(__name__)


class LogPersonReviewAdditionalSerializer(serializers.PersonReviewExtendedSerializer):

    fields = {
        'id',
        F(
            'review',
            source='review_*',
            complex=serializers.ReviewSerializer,
            complex_fields=serializers.ReviewSerializer.default_fields,
        ),
    }

    default_fields = (
        'id',
        'review',
    )


class PersonLogView(views.View):

    def process_get(self, auth, data):
        person_reviews = assemble.get_person_reviews(
            subject=auth.user,
            fields_requested=[
                FIELDS.ID,
                FIELDS.PERMISSIONS_READ,
                FIELDS.COMMENTS,
                FIELDS.CHANGES,
                FIELDS.REVIEW_ID,
                FIELDS.REVIEW_NAME,
                FIELDS.REVIEW_TYPE,
                FIELDS.REVIEW_START_DATE,
                FIELDS.REVIEW_FINISH_DATE,
            ],
            filters_chosen={
                const.FILTERS.PERSONS: [data['login']],
            },
            role_types=const.ROLE.PERSON_REVIEW_LIST_RELATED - {
                const.ROLE.PERSON.SELF,
            } | const.ROLE.CALIBRATION.ALL,
        )

        comments = []
        changes = []
        for person_review in person_reviews:
            comments.extend(person_review.comments)
            changes.extend(person_review.changes)
            if len(person_review.comments) == 0:
                log.warning(
                    'PersonLog: person_review_id: %d has no comments',
                    person_review.id
                )

        comments_serialized = serializers.LogCommentSerializer.serialize_many(comments)
        changes_serialized = serializers.LogChangeSerializer.serialize_many(changes)
        person_reviews_serialized = {
            person_review.id: LogPersonReviewAdditionalSerializer.serialize(person_review)
            for person_review in person_reviews
        }

        logs = sorted(
            comments_serialized + changes_serialized,
            key=operator.itemgetter('created_at'),
            reverse=True,
        )
        for item in logs:
            item.update(
                person_review=person_reviews_serialized[item['person_review_id']]
            )
        return {
            'log': logs,
        }
