# coding: utf-8
import operator

from django import forms
from django.views.decorators import csrf
from django.utils.decorators import method_decorator

from review.frontend import forms_common
from review.core.logic.assemble import permissions_review
from review.core.logic import assemble
from review.core.logic import review_actions
from review.lib import views
from review.shortcuts import const
from review.shortcuts import serializers
from review.shortcuts import models


FIELDS = const.FIELDS
FILTERS = const.FILTERS


class ReviewSubjectAddForm(forms.Form):

    def clean(self):
        data = self.data['data']
        result = []
        for datum in data:
            form = {
                'person': ReviewSubjectAddPersonForm,
                'department': ReviewSubjectAddDepartmentForm,
            }[datum['type']]
            result.append(form(data=datum))
        errors = [subject.errors for subject in result if not subject.is_valid()]
        if errors:
            self._errors = {'data': errors}
        return {'data': [subject.cleaned_data for subject in result]}


class ReviewSubjectAddBaseForm(forms.Form):
    structure_change = forms.ModelChoiceField(
        queryset=models.StaffStructureChange.objects.all(),
        required=True,
    )


class ReviewSubjectAddPersonForm(ReviewSubjectAddBaseForm):
    login = forms.ModelChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
        required=True,
    )

    def clean(self):
        raw_data = super(ReviewSubjectAddPersonForm, self).clean()
        if not ('login' in raw_data and 'structure_change' in raw_data):
            return {}
        return review_actions.ReviewSubject(
            type='person',
            person=raw_data['login'],
            structure_change=raw_data['structure_change'],
        )


class ReviewSubjectAddDepartmentForm(ReviewSubjectAddBaseForm):
    slug = forms.ModelChoiceField(
        queryset=models.Department.objects.all(),
        to_field_name='slug',
        required=True,
    )

    def clean(self):
        raw_data = super(ReviewSubjectAddDepartmentForm, self).clean()
        if not ('slug' in raw_data and 'structure_change' in raw_data):
            return {}
        return review_actions.ReviewSubject(
            type='department',
            department=raw_data['slug'],
            structure_change=raw_data['structure_change'],
        )


class ReviewSubjectDelForm(forms.Form):
    person_reviews = forms.ModelMultipleChoiceField(
        queryset=models.PersonReview.objects.all(),
        to_field_name='id',
        required=True,
    )


class HistoryForm(forms_common.BaseParams):
    person = forms.ModelChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
        required=True,
    )


@method_decorator(csrf.csrf_exempt, name='dispatch')
class PersonReviewsModeReviewView(views.View):
    ROLES = const.ROLE.PERSON_REVIEW_LIST_RELATED

    form_cls_get = form_cls_post = forms_common.PersonReviewsParams
    serializer = serializers.PersonReviewExtendedSerializer

    def process_get(self, auth, data):
        fields = self.get_fields(data)
        if data.get(FILTERS.REVIEWS):
            # HACK: speed up for reviews without marks
            reviews = assemble.get_reviews(
                auth.user,
                {'ids': data.get(FILTERS.REVIEWS)},
                const.FIELDS.DB_REVIEW_FIELDS | {'mark_mode'},
            )
            no_marks = all(
                review.mark_mode == const.REVIEW_MODE.MODE_DISABLED
                for review in reviews
            )
            if no_marks and const.FIELDS.MARK_LEVEL_HISTORY in fields:
                fields.remove(const.FIELDS.MARK_LEVEL_HISTORY)
        person_reviews = assemble.get_person_reviews(
            subject=auth.user,
            filters_chosen=data,
            fields_requested=fields,
            role_types=self.ROLES,
        )
        return {
            'subject': auth.user.login,
            'params': serializers.serialize_params(data),
            'person_reviews': self.serializer.serialize_many(
                objects=person_reviews,
                fields_requested=fields,
            )
        }

    def get_fields(self, data):
        fields = data.pop('fields', None) or FIELDS.DEFAULT_FOR_MODE_REVIEW
        fields = set(fields) | FIELDS.ALL_ACTION_FIELDS
        return fields

    process_post = process_get


@method_decorator(csrf.csrf_exempt, name='dispatch')
class PersonReviewsModeReviewIdsOnlyView(views.View):
    ROLES = const.ROLE.PERSON_REVIEW_LIST_RELATED

    form_cls_get = form_cls_post = forms_common.PersonReviewsParams
    serializer = serializers.PersonReviewExtendedSerializer

    def process_get(self, auth, data):
        if set(data) - FILTERS.FOR_COLLECT:
            fields = {'id', 'reviewers'}
            person_reviews = assemble.get_person_reviews(
                subject=auth.user,
                filters_chosen=data,
                fields_requested=fields,
                role_types=self.ROLES,
            )
            person_review_ids = [person_review.id for person_review in person_reviews]
        else:
            person_review_ids = list(assemble.get_person_review_ids(
                subject=auth.user,
                filters=data,
                role_types=self.ROLES,
            ))
        return {
            'subject': auth.user.login,
            'params': serializers.serialize_params(data),
            'person_review_ids': person_review_ids
        }

    process_post = process_get


class HistoryView(views.View):
    form_cls_get = HistoryForm
    roles = frozenset()

    def get_logging_context(self, data):
        return {
            'instance': "Person:{}".format(data['person'])
        }

    def process_get(self, auth, data):
        fields = data.pop('fields', None) or FIELDS.DEFAULT_FOR_REVIEWS_MODE_HISTORY
        person_reviews = assemble.get_person_reviews(
            subject=auth.user,
            filters_chosen={FILTERS.PERSONS: [data['person']]},
            fields_requested=fields,
            role_types=self.roles,
        )
        result = {
            'filters': serializers.serialize_params(data),
        }
        result.update(person_reviews_history(person_reviews, fields))
        return result

    process_post = process_get


class PersonReviewsHistoryView(HistoryView):
    # TODO: when frontend will change url - remove calibration roles
    roles = const.ROLE.PERSON_REVIEW_LIST_RELATED | {
        const.ROLE.CALIBRATION.ADMIN,
        const.ROLE.CALIBRATION.CALIBRATOR,
        const.ROLE.PERSON_REVIEW.CALIBRATOR_INHERITED,
        const.ROLE.PERSON_REVIEW.CALIBRATION_ADMIN_INHERITED,
        const.ROLE.PERSON_REVIEW.CALIBRATOR_ARCHIVED,
    }


class ReviewPersonReviewsView(views.View):

    form_cls_get = forms_common.PersonReviewsParams

    def process_get(self, auth, data):
        fields = data.pop('fields', None)
        if not fields:
            fields = {
                FIELDS.ID,
                FIELDS.PERSON_LOGIN,
                FIELDS.PERSON_GENDER,
                FIELDS.PERSON_FIRST_NAME,
                FIELDS.PERSON_LAST_NAME,
                FIELDS.PERSON_DEPARTMENT_NAME,
                FIELDS.PERSON_DEPARTMENT_SLUG,
                FIELDS.PERSON_DEPARTMENT_CHAIN_SLUGS,
                FIELDS.PERSON_DEPARTMENT_CHAIN_NAMES,
                FIELDS.REVIEWERS,
            } | FIELDS.ALL_ACTION_FIELDS
        data.update({
            FILTERS.REVIEWS: [data.pop('id')],
            FILTERS.REVIEW_STATUSES: const.REVIEW_STATUS.ALL,
        })
        person_reviews = assemble.get_person_reviews(
            subject=auth.user,
            filters_chosen=data,
            fields_requested=fields,
            role_types={
                const.ROLE.REVIEW.ADMIN,
                const.ROLE.REVIEW.SUPERREVIEWER,
                const.ROLE.REVIEW.ACCOMPANYING_HR,
            } | const.ROLE.DEPARTMENT.HRS
        )
        return {
            "filters": serializers.serialize_params(data),
            "person_reviews": serializers.PersonReviewExtendedSerializer.serialize_many(
                person_reviews,
                fields_requested=fields
            )
        }


class ReviewSubjectAddView(views.View):
    form_cls_post = ReviewSubjectAddForm
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'id', 'data'}

    def get_logging_context(self, data):
        return {
            'instance': "Review:{}".format(data['id'])
        }

    def process_post(self, auth, data):
        review = assemble.get_review(
            subject=auth.user,
            filters={'ids': [data['id']]}
        )
        permissions_review.ensure_has_action(
            review=review,
            actions=[const.REVIEW_ACTIONS.ADD_ANY_PERSONS, const.REVIEW_ACTIONS.ADD_PERSONS],
        )
        review_actions.add_review_subjects(
            person=auth.user,
            review=review,
            subjects=data['data']
        )
        return {}


class ReviewSubjectDelView(views.View):
    form_cls_post = ReviewSubjectDelForm
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'id', 'ids'}

    def get_logging_context(self, data):
        return {
            'instance': "Review:{}".format(data['id'])
        }

    def process_post(self, auth, data):
        review = assemble.get_review(
            subject=auth.user,
            filters={'ids': [data['id']]}
        )
        permissions_review.ensure_has_action(
            review=review,
            actions=[const.REVIEW_ACTIONS.DELETE_ANY_PERSONS, const.REVIEW_ACTIONS.ADD_PERSONS],
        )
        review_actions.delete_review_subject(auth.user, review, data['person_reviews'])
        return {}


def person_reviews_history(person_reviews, fields_requested=None):
    result = {}
    person_reviews = sorted(
        person_reviews,
        key=operator.attrgetter('review_start_date')
    )
    serialize_many = serializers.PersonReviewExtendedSerializer.serialize_many
    serialized = serialize_many(person_reviews, fields_requested)
    for item in serialized:
        person_data = item.pop('person')
        if 'person' not in result:
            result['person'] = person_data
    result['person_reviews'] = serialized
    return result
