# coding: utf-8
import operator

from django import forms
from review.lib import views
from review.shortcuts import const
from review.shortcuts import serializers
from review.core.logic import assemble
from review.staff import models as staff_models

FIELDS = const.FIELDS


class PersonActiveCalibrationsView(views.View):
    serializer = serializers.ExtendedCalibrationSerializer

    def process_get(self, auth, data):
        calibration_person_reviews = assemble.get_calibration_person_reviews(
            subject=auth.user,
            filters_chosen={
                'calibration_status': [const.CALIBRATION_STATUS.IN_PROGRESS, const.CALIBRATION_STATUS.DRAFT],
                'persons': [data['login']],
            },
            requested_person_review_fields=set(),
        )
        calibration_list = []
        if calibration_person_reviews:
            calibration_list = assemble.get_calibrations(
                subject=auth.user,
                filters={'ids': [cpr.calibration_id for cpr in calibration_person_reviews]},
            )

        return {
            "calibration_ids": [it.id for it in calibration_list],
            "filters": serializers.serialize_params(data),
            "calibrations": self.serializer.serialize_many(
                calibration_list,
                context=dict(language=auth.user.language),
            ),
        }


class PersonNamesForm(forms.Form):
    persons = forms.ModelMultipleChoiceField(
        queryset=staff_models.Person.objects.all(),
        required=True,
        to_field_name='login'
    )


class PersonNamesView(views.View):

    form_cls_get = PersonNamesForm

    def process_get(self, auth, data):
        lang = auth.user.language
        return {
            person.login: {
                'name': getattr(person, 'first_name_' + lang, 'name'),
                'surname': getattr(person, 'last_name_' + lang, 'surname'),
            }
            for person in list(data['persons'])
        }
