# coding: utf-8
import datetime

import logging
from datetime import date, timedelta

from django import forms
from django.core import exceptions
from django.http.response import HttpResponseForbidden
import yenv

from review.core import tasks
from review.core.logic import (
    assemble,
    domain_objs,
    review_actions,
    roles,
)
from review.core.logic.assemble import permissions_review
from review.lib import forms as lib_forms
from review.lib import serializers as lib_serializers
from review.lib import views
from review.lib import errors
from review.lib import datetimes
from review.shortcuts import const
from review.shortcuts import models
from review.shortcuts import serializers


log = logging.getLogger(__name__)


# forms
class ReviewListForm(forms.Form):
    statuses = lib_forms.VerboseMultipleChoiceField(
        choices=const.REVIEW_STATUS.CHOICES,
        required=False
    )
    fields = lib_forms.VerboseMultipleChoiceField(
        choices=[(x, x) for x in const.FIELDS.DB_REVIEW_ALL_FIELDS],
        required=False,
    )

    def clean(self):
        cleaned = super(ReviewListForm, self).clean()
        cleaned['fields'] = set(cleaned['fields']) | const.FIELDS.DB_REVIEW_WITH_PARAMETERS
        return cleaned


class CreateReviewForm(forms.Form):
    REVIEW_MODE = const.REVIEW_MODE

    MARK_MODE_SELECT = REVIEW_MODE.FORM_CHOICE_FOR_MARK_MODE

    MARK_MODE = {
        REVIEW_MODE.FORM_MARK_MODE_DISABLED: (REVIEW_MODE.MODE_DISABLED, REVIEW_MODE.MODE_DISABLED),
        REVIEW_MODE.FORM_MARK_MODE_GOLDSTAR_DISABLED: (REVIEW_MODE.MODE_MANUAL, REVIEW_MODE.MODE_DISABLED),
        REVIEW_MODE.FORM_MARK_MODE_GOLDSTAR_ENABLED: (REVIEW_MODE.MODE_MANUAL, REVIEW_MODE.MODE_MANUAL),
        REVIEW_MODE.FORM_MARK_MODE_GOLDSTAR_RESTRICTED: (REVIEW_MODE.MODE_MANUAL, REVIEW_MODE.MODE_MANUAL_BY_CHOSEN),
    }

    name = forms.CharField()
    type = lib_forms.VerboseChoiceField(
        choices=const.REVIEW_TYPE.CHOICES,
    )
    bonus_type = lib_forms.VerboseChoiceField(
        choices=const.REVIEW_BONUS_TYPE.CHOICES,
        required=False,
    )
    bonus_reason = forms.CharField(required=False)
    # dates
    start_date = forms.DateField()
    finish_date = forms.DateField()
    finish_feedback_date = forms.DateField(required=False)
    finish_submission_date = forms.DateField(required=False)
    finish_calibration_date = forms.DateField(required=False)
    finish_approval_date = forms.DateField(required=False)
    evaluation_from_date = forms.DateField(required=False)
    evaluation_to_date = forms.DateField(required=False)
    feedback_from_date = forms.DateField(required=False)
    feedback_to_date = forms.DateField(required=False)
    salary_date = forms.DateField(required=False)
    include_dismissed_after_date = forms.DateField(required=False)
    # settings
    goodies = forms.FileField(required=False)
    mark_mode = forms.ChoiceField(choices=[(x, x) for x in MARK_MODE_SELECT])
    level_change_mode = lib_forms.VerboseChoiceField(
        choices=REVIEW_MODE.CHOICES_FOR_GOODIE
    )
    salary_change_mode = lib_forms.VerboseChoiceField(
        choices=REVIEW_MODE.CHOICES_FOR_GOODIE
    )
    bonus_mode = lib_forms.VerboseChoiceField(
        choices=REVIEW_MODE.CHOICES_FOR_GOODIE
    )
    options_rsu_mode = lib_forms.VerboseChoiceField(
        choices=REVIEW_MODE.CHOICES_FOR_GOODIE
    )
    options_rsu_unit = lib_forms.VerboseChoiceField(
        choices=const.REVIEW_OPTIONS_RSU_UNIT.CHOICES,
        required=False,
    )
    deferred_payment_mode = lib_forms.VerboseChoiceField(
        choices=REVIEW_MODE.CHOICES_FOR_GOODIE,
        required=False,
    )
    scale = forms.ModelChoiceField(
        queryset=models.MarksScale.objects.all(),
        required=False,
    )

    notify_reminder_type = lib_forms.VerboseChoiceField(
        choices={
            field: field
            for field in const.REVIEW_NOTIFICATION_SETTINGS.ALL
        },
        required=False,
    )
    notify_reminder_days = lib_forms.VerboseMultipleChoiceField(
        choices=[(x, x) for x in const.WEEK_DAYS],
        required=False,
    )
    notify_reminder_date_from = forms.DateField(required=False)
    notify_reminder_date_to = forms.DateField(required=False)
    notify_events_other = lib_forms.NiceNullBooleanField(required=False)
    notify_events_superreviewer = lib_forms.NiceNullBooleanField(required=False)

    # admins
    admins = forms.ModelMultipleChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
        required=False,
    )
    super_reviewers = forms.ModelMultipleChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
        required=False,
    )
    accompanying_hrs = forms.ModelMultipleChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
        required=False,
    )

    def clean_bonus_mode(self):
        cd = self.cleaned_data

        if cd['type'] == const.REVIEW_TYPE.BONUS:
            return const.REVIEW_MODE.MODE_MANUAL

        return cd['bonus_mode']

    def clean(self):
        raw_cleaned_data = super(CreateReviewForm, self).clean()
        filters = {}
        for key, value in raw_cleaned_data.items():
            if key.startswith('finish') and value:
                start_date = raw_cleaned_data['start_date']
                if value < start_date:
                    err_msg = 'date {} must be after start date {}'
                    self.add_error(key, err_msg.format(value, start_date))
                    continue
            if key == 'mark_mode':
                mark, goldstar = CreateReviewForm.MARK_MODE[value]
                filters['mark_mode'] = mark
                filters['goldstar_mode'] = goldstar
                continue
            if key == 'goodies' and value is not None:
                value = lib_serializers.serialize_file_sheet(
                    value,
                    BonusFileForm,
                )
            if key == 'notify_reminder_days':
                value = ','.join(value)
            if key == 'notify_reminder_type':
                if value is None:
                    value = const.REVIEW_NOTIFICATION_SETTINGS.NO
                notify_reminder_days = raw_cleaned_data.get('notify_reminder_days')
                notify_reminder_date_from = raw_cleaned_data.get('notify_reminder_date_from')
                notify_reminder_date_to = raw_cleaned_data.get('notify_reminder_date_to')

                if all((
                    value == const.REVIEW_NOTIFICATION_SETTINGS.NO,
                    not notify_reminder_days,
                    notify_reminder_date_from is None,
                    notify_reminder_date_to is None,
                )):
                    continue
                elif all((
                    value == const.REVIEW_NOTIFICATION_SETTINGS.ONCE,
                    not notify_reminder_days,
                    notify_reminder_date_from is not None,
                    notify_reminder_date_to is None,
                )):
                    continue
                elif all((
                    value == const.REVIEW_NOTIFICATION_SETTINGS.PERIOD,
                    notify_reminder_days,
                    notify_reminder_date_from is not None,
                    notify_reminder_date_to is not None,
                )):
                    continue
                else:
                    self.add_error(
                        field='notify_reminder_type',
                        error="Incorrect notification settings for type %s" % value
                    )
                    raise exceptions.ValidationError(self.errors)
            filters[key] = value
        if not filters.get('feedback_to_date'):
            filters['feedback_to_date'] = filters['start_date']
        if not filters.get('feedback_from_date'):
            filters['feedback_from_date'] = datetimes.shifted(
                filters['feedback_to_date'],
                months=-6,
            )
        return filters


LOG_FIELDS = {
    'name',
    'type',
    'bonus_type',
    'bonus_reason',
    'start_date',
    'finish_date',
    'finish_feedback_date',
    'finish_submission_date',
    'finish_calibration_date',
    'finish_approval_date',
    'evaluation_from_date',
    'evaluation_to_date',
    'feedback_from_date',
    'feedback_to_date',
    'salary_date',
    'include_dismissed_after_date',
    'mark_mode',
    'level_change_mode',
    'salary_change_mode',
    'bonus_mode',
    'options_rsu_mode',
    'options_rsu_unit',
    'deferred_payment_mode',
    'scale',
    'notify_reminder_type',
    'notify_reminder_days',
    'notify_reminder_date_from',
    'notify_reminder_date_to',
    'notify_events_other',
    'notify_events_superreviewer',
    'admins',
    'super_reviewers',
    'accompanying_hrs',
}


class BonusFileForm(forms.Form):
    level = forms.IntegerField(
        min_value=0,
        max_value=const.VALIDATION.LEVEL_CHANGE_MAX,
    )
    mark = forms.CharField(
        initial=const.MARK.NOT_SET,
    )
    goldstar = lib_forms.VerboseChoiceField(choices=const.GOLDSTAR.VERBOSE)
    level_change = forms.IntegerField(
        min_value=const.VALIDATION.LEVEL_CHANGE_MIN,
        max_value=const.VALIDATION.LEVEL_CHANGE_MAX
    )
    salary_change = forms.DecimalField(
        max_digits=7,
        decimal_places=2,
        min_value=const.VALIDATION.SALARY_CHANGE_MIN,
        max_value=const.VALIDATION.SALARY_CHANGE_MAX,
    )
    bonus = forms.DecimalField(
        max_digits=7,
        decimal_places=2,
        min_value=const.VALIDATION.BONUS_MIN,
        max_value=const.VALIDATION.BONUS_MAX,
    )
    options_rsu = forms.IntegerField(
        min_value=const.VALIDATION.OPTIONS_RSU_MIN,
        max_value=const.VALIDATION.OPTIONS_RSU_MAX
    )

    def __init__(self, data, review=None, **kwargs):
        super(BonusFileForm, self).__init__(data=data, **kwargs)


class ReviewWorkflowForm(forms.Form):
    workflow = lib_forms.VerboseChoiceField(
        choices=[(x, x) for x in const.REVIEW_ACTIONS.STATUS]
    )


# views
class ReviewListView(views.View):
    form_cls_get = ReviewListForm
    form_cls_post = CreateReviewForm
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'id', 'persons'}

    def process_post(self, auth, data):
        if not roles.has_global_roles(
            person=auth.user,
            global_roles=[const.ROLE.GLOBAL.REVIEW_CREATOR]
        ):
            raise errors.PermissionDenied(
                type='review',
                action='create',
            )
        data['author'] = auth.user
        review = review_actions.create_review(auth.user, data)
        review_actions.add_global_admins(review.id)
        return {'id': review.id}

    def process_get(self, auth, data):
        fields = data['fields'] | {'actions'}
        reviews = assemble.get_reviews(
            subject=auth.user,
            filters=data,
            requested_fields=fields,
        )
        reviews = domain_objs.sort_reviews_by_importance(reviews)
        most_relevant = domain_objs.get_most_relevant_review(reviews)
        return {
            'filters': serializers.serialize_params(data),
            'reviews': serializers.ExtendedReviewSerializer.serialize_many(
                objects=reviews,
                fields_requested=fields,
            ),
            'most_relevant_id': most_relevant.id if most_relevant else None,
        }


class ReviewDetailView(views.View):
    form_cls_post = CreateReviewForm

    def get_logging_context(self, data):
        return {
            'instance': "Review:{}".format(data['id'])
        }

    def process_post(self, auth, data):
        review = assemble.get_review(
            subject=auth.user,
            filters={'ids': [data['id']]},
            requested_fields=const.FIELDS.DB_REVIEW_ALL_FIELDS
        )
        permissions_review.ensure_has_action(
            review=review,
            actions=[const.REVIEW_ACTIONS.EDIT],
        )
        review = review_actions.edit_review(review, data)
        roles.async_denormalize_person_review_roles(review_id=review.id)
        return {'id': review.id}

    def process_get(self, auth, data):
        review_id = data['id']
        review = assemble.get_review(
            subject=auth.user,
            filters={
                'ids': [review_id],
            },
            requested_fields=const.FIELDS.DB_REVIEW_ALL_FIELDS,
        )
        if review is None:
            exists = models.Review.objects.filter(id=review_id).exists()
            err_cls = errors.PermissionDenied if exists else errors.NotFound
            raise err_cls(
                type='review',
                id=review_id,
            )
        if review.actions[const.REVIEW_ACTIONS.LIST_GOODIES] == const.OK:
            review.goodies = assemble.get_review_goodies(auth.user, review.id)

        fields_requested = const.FIELDS.DB_REVIEW_ALL_FIELDS | {
            'notify_reminder_type',
            'actions',
            'goodies',
        }
        return serializers.ExtendedReviewSerializer.serialize(
            obj=review,
            fields_requested=fields_requested,
        )


class ReviewFollowWorkflowView(views.View):
    form_cls_post = ReviewWorkflowForm

    def get_logging_context(self, data):
        return {
            'instance': "Review:{}".format(data['id'])
        }

    def process_post(self, auth, data):
        review = assemble.get_review(
            subject=auth.user,
            filters={'ids': [data['id']]}
        )
        permissions_review.ensure_has_action(
            review=review,
            actions=[data['workflow']],
        )
        review_actions.validate_transition(
            review=review,
            transition=data['workflow'],
        )

        from review.core import tasks
        func = tasks.follow_workflow
        if yenv.type != 'development':
            func = func.delay
        func(user_login=auth.user.login, review_id=review.id, new_workflow=data['workflow'])

        return {"id": review.id}


class ReviewLoadKpi(views.View):

    def get_logging_context(self, data):
        return {
            'instance': "Review:{}".format(data['id'])
        }

    def process_post(self, auth, data):
        review = assemble.get_review(
            subject=auth.user,
            filters={'ids': [data['id']]},
            requested_fields=const.FIELDS.DB_REVIEW_WITH_PARAMETERS | {'evaluation_from_date', 'evaluation_to_date'},
        )
        permissions_review.ensure_has_action(
            review=review,
            actions=[const.REVIEW_ACTIONS.LOAD_KPI],
        )
        date_format = '%d.%m.%Y'
        date_from = review.evaluation_from_date.strftime(date_format)
        date_to = review.evaluation_to_date.strftime(date_format)

        from review.core import tasks
        func = tasks.load_kpi
        if yenv.type != 'development':
            func = func.delay
        func(review.id, date_from, date_to)

        return {"id": review.id}


class UpdateProductStructureForm(forms.Form):
    days_before = forms.IntegerField(required=False)
    days_after = forms.IntegerField(required=False)

    def clean(self):
        res = super(UpdateProductStructureForm, self).clean()
        res['days_before'] = res['days_before'] or 100
        res['days_after'] = res['days_after'] or 10
        return res


class UpdateProductStructureView(views.View):
    form_cls_post = UpdateProductStructureForm

    def process_post(self, auth, data):
        review_id = data['id']
        review = assemble.get_review(
            auth.user,
            {'ids': [review_id]},
            requested_fields=const.FIELDS.DB_REVIEW_WITH_PARAMETERS | {'evaluation_to_date'},
        )
        if review.actions[const.REVIEW_ACTIONS.LOAD_PRODUCT_SCHEME] != const.OK:
            return HttpResponseForbidden()
        date_from = review.evaluation_to_date - timedelta(days=data['days_before'])
        date_from = date_from.strftime('%d.%m.%Y')
        date_to = review.evaluation_to_date + timedelta(days=data['days_after'])
        date_to = date_to.strftime('%d.%m.%Y')
        log.info('User {} started product sync for review {} on dates from {} to {}'.format(
            auth.user.login,
            review_id,
            date_from,
            date_to,
        ))
        review_actions.load_product_scheme(review_id, date_from, date_to)
        return {
            'status': 'ok',
            'used_dates': {
                'date_from': date_from,
                'date_to': date_to,
            }
        }
