# coding: utf-8
from django import forms
from django.core import exceptions
from jsonfield.fields import JSONFormField

from review.frontend import forms_common
from review.lib import views
from review.shortcuts import models


class StoredFilterForm(forms.Form):
    name = forms.CharField()
    value = JSONFormField()

    def clean_value(self):
        filter_params = self.cleaned_data['value']
        if not filter_params:
            raise exceptions.ValidationError("Filter shouldn't be empty")

        filter_params_form = forms_common.PersonReviewsParams(data=filter_params)
        if not filter_params_form.is_valid():
            self._errors['value'] = [filter_params_form.errors]
        return filter_params


class StoredFiltersView(views.View):
    form_cls_post = StoredFilterForm
    log_params_for = {'post'}
    WHITE_LIST_TO_LOG = {'name', 'value'}

    def process_post(self, auth, data):
        created_filter = models.StoredFilter.objects.create(
            owner=auth.user,
            name=data['name'],
            value=data['value']
        )
        return {'id': created_filter.id}

    def process_get(self, auth, data):
        saved_filters = models.StoredFilter.objects.filter(
            owner=auth.user
        )
        return serialize_saved_filters(saved_filters)


class StoredFiltersDetailView(views.View):
    form_cls_post = StoredFilterForm
    form_cls_delete = None
    log_params_for = {'post', 'delete'}
    WHITE_LIST_TO_LOG = {'id', 'name', 'value'}

    def process_post(self, auth, data):
        models.StoredFilter.objects.filter(
            owner=auth.user,
            id=data['id']
        ).update(
            name=data['name'],
            value=data['value']
        )
        return {}

    def process_delete(self, auth, data):
        models.StoredFilter.objects.filter(
            owner=auth.user,
            id=data['id']
        ).delete()
        return {}


def serialize_saved_filters(saved_filters):
    return {
        "saved_filters": [
            {
                "id": _filter.id,
                "name": _filter.name,
                "value": _filter.value
            }
            for _filter in saved_filters
        ]
    }
