# coding: utf-8
from django.middleware import csrf
from waffle import flag_is_active
from waffle.compat import cache
from waffle.models import Flag
from waffle.utils import get_setting, keyfmt

from review.core.logic import permissions
from review.lib import views
from review.shortcuts import const
from review.staff.logic import get_is_chief_by_login, get_hr_roles_by_login


class UserView(views.View):

    def get(self, request, *args, **kwargs):
        auth = request.auth
        language = auth.user.language
        user_login = auth.user.login

        is_chief = get_is_chief_by_login(user_login)
        hr_roles = get_hr_roles_by_login(user_login)
        is_hranalyst = const.STAFF_ROLE.HR.HR_ANALYST in hr_roles
        is_hrbp = const.STAFF_ROLE.HR.HR_PARTNER in hr_roles

        return {
            'login': user_login,
            'first_name': getattr(auth.user, 'first_name_' + language),
            'last_name': getattr(auth.user, 'last_name_' + language),
            'is_dismissed': auth.user.is_dismissed,
            'actions': permissions.get_global_actions(auth.user),
            'gender': const.GENDER.VERBOSE[auth.user.gender],
            'language': language,
            'is_chief': is_chief,
            'is_hranalyst': is_hranalyst,
            'is_hrbp': is_hrbp,
            'csrftoken': csrf.get_token(request=self.request),
            'marks_format_mode': auth.user.marks_format_mode,
            'flags': self.get_waffle_flags(request),
        }

    def get_waffle_flags(self, request):
        # Note: code copied from waffle.views._generate_waffle_js
        flags = cache.get(keyfmt(get_setting('ALL_FLAGS_CACHE_KEY')))
        if flags is None:
            flags = Flag.objects.values_list('name', flat=True)
            cache.add(keyfmt(get_setting('ALL_FLAGS_CACHE_KEY')), flags)
        return {f: flag_is_active(request, f) for f in flags}
