from collections import defaultdict
from logging import getLogger

from django.utils import timezone
from ylog.context import log_context

from review.core import const
from review.core.logic import assemble
from review.core.logic.bulk import diffs, save
from review.core.models import Review
from review.gradient.models import UmbrellaPerson, MainProductPerson, Umbrella
from review.lib import helpers
from review.staff.models import Person

log = getLogger(__name__)


MAIN_UMBRELLA_ENGAGEMENT_THRESHOLD = 50


@helpers.timeit_no_args_logging
def freeze_gradient_data(review_id, freezing_datetime=None, person_review_ids=None):
    res = {}
    freezing_datetime = freezing_datetime or timezone.now()
    robot = Person.objects.get(login='robot-review')

    with log_context(review_id=review_id):
        if not need_freeze_gradient_data(review_id):
            log.info('Gradient: skip gradient data freezing')
            return res

        Review.objects.filter(id=review_id).update(product_schema_loaded=freezing_datetime)

        if person_review_ids:
            filters_chosen = {const.FILTERS.IDS: person_review_ids}
        else:
            filters_chosen = {const.FILTERS.REVIEWS: [review_id]}

        person_reviews = assemble.get_person_reviews(
            subject=robot,
            fields_requested=const.FIELDS.FOR_ROBOT,
            filters_chosen=filters_chosen,
        )

        person_ids = [pr.person_id for pr in person_reviews]
        person_umbrellas = get_person_umbrellas_for_persons(person_ids, freezing_datetime)
        person_main_products = get_main_products_for_persons(person_ids, freezing_datetime)
        default_umbrella = Umbrella.objects.filter(main_product__isnull=True).order_by('id').first()

        for person_review in person_reviews:
            with log_context(person_review_id=person_review.id, person_id=person_review.person_id):
                main_umbrella = choose_main_umbrella(
                    umbrella_person_list=person_umbrellas.get(person_review.person_id, []),
                    default_umbrella=default_umbrella,
                )
                main_product = person_main_products.get(person_review.person_id)

                params = {
                    const.PERSON_REVIEW_ACTIONS.UMBRELLA: main_umbrella or -1,
                    const.PERSON_REVIEW_ACTIONS.MAIN_PRODUCT: main_product or -1,
                }
                res[person_review] = diffs.make_diff_for_actions(
                    subject=robot,
                    person_review=person_review,
                    params=params,
                )

        save.save_for_same_action_set(
            subject=robot,
            diff_bulk={
                person_review: actions_result['changes']
                for person_review, actions_result in res.items()
            },
            subject_type=const.PERSON_REVIEW_CHANGE_TYPE.ROBOT,
        )

    return res


@helpers.timeit_no_args_logging
def need_freeze_gradient_data(review_id, freezing_datetime=None):
    active_umbrellas_exists = (
        UmbrellaPerson.objects
        .filter_active(freezing_datetime)
        .filter(person__person_review__review_id=review_id)
        .exists()
    )
    active_main_products_exists = (
        MainProductPerson.objects
        .filter_active(freezing_datetime)
        .filter(person__person_review__review_id=review_id)
        .exists()
    )
    return active_umbrellas_exists or active_main_products_exists


@helpers.timeit_no_args_logging
def get_person_umbrellas_for_persons(person_ids, freezing_datetime=None):
    umbrella_persons = (
        UmbrellaPerson.objects
        .filter_active(freezing_datetime)
        .filter(person__in=person_ids)
        .select_related('umbrella')
        .order_by('id')
    )
    result = defaultdict(list)
    for up in umbrella_persons:
        result[up.person_id].append(up)
    return result


@helpers.timeit_no_args_logging
def get_main_products_for_persons(person_ids, freezing_datetime=None):
    main_product_persons = (
        MainProductPerson.objects
        .filter(person__in=person_ids)
        .filter_active(freezing_datetime)
        .select_related('main_product')
        .order_by('-id')
    )
    return {mp.person_id: mp.main_product for mp in main_product_persons}


@helpers.timeit_no_args_logging
def choose_main_umbrella(umbrella_person_list, default_umbrella=None):
    main_umbrella = None
    main_umbrellas_candidates = [
        u for u in umbrella_person_list
        if u.engagement > MAIN_UMBRELLA_ENGAGEMENT_THRESHOLD
    ]

    if len(main_umbrellas_candidates) > 0:
        main_umbrella = main_umbrellas_candidates[0].umbrella
        if len(main_umbrellas_candidates) > 1:
            log.warning(
                'Gradient: got several umbrellas with more than %s engagement, umbrellas_id=%s',
                MAIN_UMBRELLA_ENGAGEMENT_THRESHOLD,
                ','.join(str(up.umbrella_id) for up in main_umbrellas_candidates)
            )
    elif len(umbrella_person_list) > 0:
        log.info('Gradient: use default_umbrella=%s', default_umbrella)
        main_umbrella = default_umbrella

    if main_umbrella is None:
        log.warning('Gradient: could not choose main umbrella')

    return main_umbrella
