from django.db import models
from django.utils import timezone

from review.core.models import Review, PersonReview
from review.staff.models import Person


class MainProduct(models.Model):
    """
    Also known as Value Stream
    """
    issue_key = models.CharField(max_length=255, unique=True, null=True, blank=True)
    name = models.CharField(max_length=255)
    abc_service_id = models.IntegerField(null=True, unique=True, blank=True)

    class Meta(object):
        db_table = 'core_mainproduct'

    def __str__(self):
        return f'{self.abc_service_id}: {self.name}'


class Umbrella(models.Model):
    issue_key = models.CharField(max_length=255, unique=True, null=True)
    name = models.CharField(max_length=255)
    main_product = models.ForeignKey(
        to=MainProduct,
        related_name='umbrellas',
        related_query_name='umbrella',
        null=True,
        blank=True,
    )

    class Meta(object):
        db_table = 'core_umbrella'

    def __str__(self):
        return f'{self.issue_key}: {self.name}'


class MainProductReview(models.Model):
    """
    Legacy. It was used in 20-21 year's review and keeps historical data.
    Now MainProduct and Review are not relate at all.
    """
    main_product = models.ForeignKey(
        to=MainProduct,
        related_name='main_product_reviews',
        related_query_name='main_product_review',
    )
    review = models.ForeignKey(
        to=Review,
        related_name='main_product_reviews',
        related_query_name='main_product_review',
    )

    class Meta(object):
        unique_together = ('review', 'main_product')
        db_table = 'core_mainproductreview'


class UmbrellaReview(models.Model):
    """
    Legacy. It was used in 20-21 year's review and keeps historical data.
    Now Umbrella and Review are not relate at all.
    """
    umbrella = models.ForeignKey(
        to=Umbrella,
        related_name='umbrella_reviews',
        related_query_name='umbrella_review',
    )
    review = models.ForeignKey(
        to=Review,
        related_name='umbrella_reviews',
        related_query_name='umbrella_review',
    )

    class Meta(object):
        unique_together = ('review', 'umbrella')
        db_table = 'core_umbrellareview'


class UmbrellaPersonReview(models.Model):
    # FIXME legacy, data was copied to PersonReview.umbrella field,
    # so this model is supposed to be deleted
    umbrella = models.ForeignKey(
        to=Umbrella,
        related_name='umbrella_person_reviews',
        related_query_name='umbrella_person_review',
    )
    person_review = models.ForeignKey(
        to=PersonReview,
        related_name='umbrella_person_reviews',
        related_query_name='umbrella_person_review',
    )

    class Meta(object):
        unique_together = ('person_review', 'umbrella')
        db_table = 'core_umbrellapersonreview'


class EngagedQueryset(models.QuerySet):

    def filter_active(self, active_datetime=None):
        active_datetime = active_datetime or timezone.now()
        return self.filter(
            models.Q(engaged_from__lte=active_datetime) | models.Q(engaged_from__isnull=True),
            models.Q(engaged_to__gte=active_datetime) | models.Q(engaged_to__isnull=True),
        )


class MainProductPerson(models.Model):
    main_product = models.ForeignKey(
        to=MainProduct,
        related_name='main_product_persons',
        related_query_name='main_product_person',
    )
    person = models.ForeignKey(
        to=Person,
        related_name='main_product_persons',
        related_query_name='main_product_person',
    )
    engaged_from = models.DateTimeField(null=True, blank=True)
    engaged_to = models.DateTimeField(null=True, blank=True)

    objects = EngagedQueryset.as_manager()


class UmbrellaPerson(models.Model):
    umbrella = models.ForeignKey(
        to=Umbrella,
        related_name='umbrella_persons',
        related_query_name='umbrella_person',
    )
    person = models.ForeignKey(
        to=Person,
        related_name='umbrella_persons',
        related_query_name='umbrella_person',
    )
    engagement = models.DecimalField(max_digits=16, decimal_places=3)
    engaged_from = models.DateTimeField(null=True, blank=True)
    engaged_to = models.DateTimeField(null=True, blank=True)

    objects = EngagedQueryset.as_manager()
