import datetime as dt
import logging
from collections import defaultdict
from typing import Iterator, Dict, List, Tuple

import attr
import more_itertools
import yenv
from dateutil import parser
from django.db import transaction
from django.conf import settings

from review.gradient import models as gradient_models
from review.lib import lock, datetimes, helpers
from review.staff import (
    const as staff_const,
    models as staff_models,
)
from review.staff.sync.connector import StaffConnector


log = logging.getLogger(__name__)


@lock.get_lock_or_do_nothing_task
@helpers.timeit_no_args_logging
@transaction.atomic
def sync_vs_and_umbrellas():
    # value stream and main product are the same entities
    connector = StaffConnector()
    umbrellas_from_db = {}
    for umbrella in gradient_models.Umbrella.objects.all():
        umbrellas_from_db[umbrella.issue_key] = umbrella
    main_products_from_db = {
        it.abc_service_id: it
        for it in (
            gradient_models.MainProduct.objects
            .filter(abc_service_id__isnull=False)
            # there is some duplicated abc_service_id in old data
            .order_by('id')
        )
    }

    _sync_umbrella_structure(
        connector,
        # mutable by function
        umbrellas_from_db,
        main_products_from_db,
    )

    login_to_id = _get_persons_from_db()
    _sync_person_umbrella_engagement(
        connector,
        login_to_id,
        umbrellas_from_db,
    )
    _sync_person_main_product_engagement(
        connector,
        login_to_id,
        main_products_from_db,
    )


@helpers.timeit_no_args_logging
def _sync_person_main_product_engagement(
    connector: StaffConnector,
    login_to_id: Dict[str, int],
    abc_id_to_mp: Dict[int, gradient_models.MainProduct],
):
    engagements_from_db = (
        gradient_models.MainProductPerson.objects
        .filter_active()
        .select_related('person')
    )
    login_to_engagemet = {
        it.person.login: it
        for it in engagements_from_db
    }
    now = datetimes.now()
    staff_info = _get_person_value_stream(connector, login_to_id.keys())
    for staff_person_vs in staff_info:
        main_product = abc_id_to_mp.get(staff_person_vs.abc_service_id)
        if not main_product:
            log.info(
                'Main product %s is synced during person vs engagement',
                staff_person_vs.abc_service_id,
            )
            main_product = _create_main_product(
                name=staff_person_vs.name,
                abc_service_id=staff_person_vs.abc_service_id,
            )
            abc_id_to_mp[main_product.abc_service_id] = main_product
        mp_engagement = login_to_engagemet.get(staff_person_vs.login)
        if not mp_engagement:
            gradient_models.MainProductPerson.objects.create(
                main_product=main_product,
                engaged_from=now,
                person_id=login_to_id[staff_person_vs.login],
            )
            log.info(
                f'Started main_product engagement for {staff_person_vs.login}'
                f' at main product {main_product.abc_service_id}'
            )
        elif mp_engagement.main_product.abc_service_id != staff_person_vs.abc_service_id:
            mp_engagement.engaged_to = now
            mp_engagement.save()
            log.info(
                f'Finished main_product engagement for {staff_person_vs.login}'
                f' at main product {main_product.abc_service_id}'
            )
            gradient_models.MainProductPerson.objects.create(
                main_product=main_product,
                engaged_from=now,
                person_id=login_to_id[staff_person_vs.login],
            )
            log.info(
                f'Started main_product engagement for {staff_person_vs.login}'
                f' at {main_product.abc_service_id}'
            )


@helpers.timeit_no_args_logging
def _sync_person_umbrella_engagement(
    connector: StaffConnector,
    login_to_id: Dict[str, int],
    umbrellas_from_db: Dict[str, gradient_models.Umbrella],
):
    engagements_from_db = (
        gradient_models.UmbrellaPerson.objects
        .filter_active()
        .select_related('person')
    )
    login_to_unique_to_engagement = defaultdict(dict)
    for model in engagements_from_db:
        unique = (
            model.umbrella_id,
            model.engaged_from and model.engaged_from.isoformat()[:19],
        )
        login_to_unique_to_engagement[model.person.login][unique] = model
    staff_info = _get_umbrella_engagement_from_staff(connector, login_to_id.keys())
    login_to_engaged_dates = defaultdict(list)
    for login, staff_engagements in staff_info:
        existing_unique_to_engagement = login_to_unique_to_engagement[login]
        engaged_dates = login_to_engaged_dates[login]
        for staff_eng in staff_engagements:
            umbrella = umbrellas_from_db.get(staff_eng.goal)
            if not umbrella:
                if yenv.type == 'production':
                    log.error('Staff vs: no umbrella %s in db', staff_eng.goal)
                continue
            unique = (
                umbrella.id,
                staff_eng.engaged_from and staff_eng.engaged_from.isoformat()[:19],
            )
            if staff_eng.engaged_from:
                engaged_dates.append(staff_eng.engaged_from)
            existing_engagement = existing_unique_to_engagement.pop(unique, None)
            if not existing_engagement:
                gradient_models.UmbrellaPerson.objects.create(
                    umbrella=umbrella,
                    person_id=login_to_id[login],
                    engagement=staff_eng.engagement,
                    engaged_from=staff_eng.engaged_from,
                    engaged_to=staff_eng.engaged_to,
                )
                log.info(
                    f'Created umbrella engagement for {login}'
                    f' at umbrella {umbrella.issue_key} with prc {staff_eng.engagement}'
                    f' from {staff_eng.engaged_from} to {staff_eng.engaged_to}'
                )
                continue
            changed = False
            if existing_engagement.engaged_to != staff_eng.engaged_to:
                log.info(
                    f'Changed umbrella engagement_to for {login}'
                    f' from {existing_engagement.engaged_to}'
                    f' to {staff_eng.engaged_to}'
                )
                existing_engagement.engaged_to = staff_eng.engaged_to
                changed = True
            if existing_engagement.engagement != staff_eng.engagement:
                existing_engagement.engagement = staff_eng.engagement
                log.warning(
                    f'Changed umbrella engagement prc for {login}'
                    f' from {existing_engagement.engagement}'
                    f' to {staff_eng.engagement}'
                )
                changed = True
            if changed:
                existing_engagement.save()
    for login, finished_engagements in login_to_unique_to_engagement.items():
        # FIXME
        # This alogorythm is guessing appropriate engaged_to date
        # It's better to get all engagements from staff, not just actual
        # Data can be easly fixed on our side just by deleting .filter_active()
        # at the start of this function and this loop
        dates = sorted(login_to_engaged_dates[login])
        for db_engagement in finished_engagements.values():
            # it's ok to use naive date here
            if db_engagement.engaged_from:
                naive_date = db_engagement.engaged_from.replace(tzinfo=None)
                finish_date = next(
                    (d for d in dates if d > naive_date),
                    None,
                )
            else:
                finish_date = dates and dates[0]
            if finish_date:
                # we can't guess any finish date for persons
                # not having actual engagements now
                db_engagement.engaged_to = finish_date
                db_engagement.save()


@helpers.timeit_no_args_logging
def _sync_umbrella_structure(
    connector: StaffConnector,
    umbrellas_from_db: Dict[str, gradient_models.Umbrella],
    main_products_from_db: Dict[int, gradient_models.MainProduct],
) -> Dict[int, gradient_models.Umbrella]:
    for staff_umbrella in _get_umberllas_from_staff(connector):
        staff_vs = staff_umbrella.value_stream
        db_main_product = main_products_from_db.get(staff_vs.abc_service_id)
        if not db_main_product:
            db_main_product = _create_main_product(
                name=staff_vs.name,
                abc_service_id=staff_vs.abc_service_id,
            )
            main_products_from_db[db_main_product.abc_service_id] = db_main_product
        elif db_main_product.name != staff_vs.name:
            log.info(
                f'Changed name for main product {db_main_product.abc_service_id}'
                f' from {db_main_product.name}'
                f' to {staff_vs.name}'
            )
            db_main_product.name = staff_vs.name
            db_main_product.save()
        db_umbrella = umbrellas_from_db.get(staff_umbrella.goal)
        if not db_umbrella:
            db_umbrella = gradient_models.Umbrella.objects.create(
                issue_key=staff_umbrella.goal,
                name=staff_umbrella.name,
                main_product_id=db_main_product.id,
            )
            umbrellas_from_db[db_umbrella.issue_key] = db_umbrella
            log.info(
                f'Created umbrella {db_umbrella.issue_key} {db_umbrella.name}'
                f' issue key {db_umbrella.issue_key}'
            )
        elif db_umbrella.name != staff_umbrella.name:
            log.info(
                f'Changed main product {db_umbrella.issue_key} name'
                f' from {db_umbrella.name}'
                f' to {staff_umbrella.name}'
            )
            db_umbrella.name = staff_umbrella.name
            db_umbrella.save()
    if not umbrellas_from_db.get(None):
        # Single umbrella without main product has to exist
        umbrellas_from_db[None] = gradient_models.Umbrella.objects.create(
            name="Работает на весь VS",
            main_product_id=None,
        )
        log.info(f'Created default umbrella with id {umbrellas_from_db[None].id}')


def _create_main_product(
    name,
    abc_service_id,
):
    res = gradient_models.MainProduct.objects.create(
        name=name,
        abc_service_id=abc_service_id,
    )
    log.info(
        f'Created main product {res.abc_service_id}'
        f' named {res.name} at abc {res.abc_service_id}'
    )
    return res


@attr.s
class StaffVS:
    name = attr.ib(type=str)
    name_en = attr.ib(type=str)
    url = attr.ib(type=str)
    abc_service_id = attr.ib(type=int)


@attr.s
class StaffUmbrella:
    goal = attr.ib(type=str)
    goal_id = attr.ib(type=int)
    name = attr.ib(type=str)
    value_stream = attr.ib(type=StaffVS)


def _get_umberllas_from_staff(connector: StaffConnector) -> Iterator[StaffUmbrella]:
    resp = connector.get(
        resource=staff_const.UMBRELLAS,
    )
    while resp.get('result'):
        for umbrella_dict in resp['result']:
            vs_info = umbrella_dict['value_stream']
            if not vs_info['abc_service_id']:
                if yenv.type == 'production':
                    log.error('Staff vs: no abc_service_id for %s', vs_info["url"])
                continue
            yield StaffUmbrella(
                umbrella_dict['goal'],
                umbrella_dict['goal_id'],
                umbrella_dict['name'],
                StaffVS(
                    vs_info['name'],
                    vs_info['name_en'],
                    vs_info['url'],
                    vs_info['abc_service_id'],
                ),
            )
        continuation_token = resp.get('continuation_token')
        if continuation_token is None:
            break
        resp = connector.get(
            resource=staff_const.UMBRELLAS,
            json={'continuation_token': continuation_token},
        )


@attr.s
class StaffUmbrellaEngagement:
    goal = attr.ib(type=str)
    engagement = attr.ib(type=str)
    engaged_from = attr.ib(type=dt.datetime)
    engaged_to = attr.ib(type=dt.datetime)


def _get_umbrella_engagement_from_staff(
    connector: StaffConnector,
    logins: List[str],
) -> Iterator[Tuple[str, List[StaffUmbrellaEngagement]]]:
    chunked = more_itertools.chunked(logins, settings.GRADIENT_SYNC_CHUNK_SIZE)
    for chunk in chunked:
        resp = connector.post(
            resource=staff_const.UMBRELLAS_ENGAGEMENT,
            json={'persons': chunk},
        )
        for login, engagements in resp.items():
            resp_objs = [
                StaffUmbrellaEngagement(
                    eng['goal'],
                    eng['engagement'],
                    parser.parse(eng['engaged_from']) if eng['engaged_from'] else None,
                    parser.parse(eng['engaged_to']) if eng['engaged_to'] else None,
                )
                for eng in engagements
            ]
            yield login, resp_objs


@attr.s
class StaffPersonVS:
    login = attr.ib(type=str)
    url = attr.ib(type=str)
    name = attr.ib(type=str)
    name_en = attr.ib(type=str)
    abc_service_id = attr.ib(type=int)


def _get_person_value_stream(
    connector: StaffConnector,
    logins: List[str],
) -> Iterator[StaffPersonVS]:
    chunked = more_itertools.chunked(logins, settings.GRADIENT_SYNC_CHUNK_SIZE)
    for chunk in chunked:
        resp = connector.post(
            resource=staff_const.VS_ENGAGEMENT,
            json={'persons': chunk},
        )
        for login, value_streams in resp.items():
            vs_info = next(
                (it for it in value_streams if 'vs' in it['service_tags']),
                None,
            )
            if not vs_info:
                continue
            if not vs_info['abc_service_id']:
                if yenv.type == 'production':
                    log.error('Staff vs: no abc_service_id for %s', vs_info["url"])
                continue
            yield StaffPersonVS(
                login,
                vs_info['url'],
                vs_info['name'],
                vs_info['name_en'],
                vs_info['abc_service_id'],
            )


def _get_persons_from_db() -> Dict[str, int]:
    return dict(
        staff_models.Person.objects
        .filter(is_dismissed=False)
        .values_list('login', 'id')
    )
