# coding: utf-8
import os
import logging

from django.contrib import auth
from django import http
from django.conf import settings
from django.utils.deprecation import MiddlewareMixin
from django_yauth.user import AnonymousYandexUser


from review.lib import (
    errors,
    helpers,
)


log = logging.getLogger(__name__)


def _is_skip_auth_path(request):
    return any(request.path.startswith(path) for path in settings.NOT_FORCE_AUTH)


def has_real_user(request):
    return (
        request.yauser is not None and
        not isinstance(request.yauser, AnonymousYandexUser)
    )


class SetUserMiddleware(MiddlewareMixin):

    def process_request(self, request):
        if not has_real_user(request) and _is_skip_auth_path(request):
            return
        user_model = auth.get_user_model()
        login = getattr(request.yauser, 'login', None)
        if login:
            filter_params = {'login': login}
        else:
            filter_params = {'uid': request.yauser.uid}
        log.info('Setting user for {} PID {}'.format(filter_params, helpers.get_pid()))
        request.user = user_model.objects.filter(**filter_params).first()
        if request.user is None:
            msg = 'Unknown user {}'.format(request.yauser.login)
            return http.HttpResponse(msg, status=401)
        if not login:
            request.yauser.fields['login'] = request.user.login
        log.info('Set user {} PID {}'.format(request.user.login, helpers.get_pid()))


class SetAuthStaffIdMiddleware(MiddlewareMixin):

    def process_request(self, request):
        user = getattr(request, 'user', None)
        auth = getattr(request, 'auth', None)
        if user is None or auth is None:
            return
        auth.staff_id = user.id


class YandexAuthBackendRequiredMiddleware(MiddlewareMixin):

    def process_request(self, request):
        if not has_real_user(request) and not _is_skip_auth_path(request):
            return http.HttpResponse('Not authenticated', status=401)
        log.info('Auth complete PID {}'.format(helpers.get_pid()))


class MoonNightMiddleware(MiddlewareMixin):

    def process_request(self, request):
        from review.werewolf.models import Werewolf

        if settings.IS_PRODUCTION_DB:
            return

        no_werewoolf = ['/admin', '/oebs']
        if any(request.path.startswith(url) for url in no_werewoolf):
            return

        if any(request.path.startswith(path) for path in settings.NOT_FORCE_AUTH):
            return

        if not has_real_user(request):
            return

        login = getattr(request.yauser, 'login', None)
        if not login:
            login = (
                auth.get_user_model().objects
                .filter(uid=getattr(request.yauser, 'uid', None))
                .values_list('login', flat=True)
                .first()
            )

        werewolf = Werewolf.objects.filter(
            login=login,
            is_active=True
        ).first()

        if werewolf:
            werewolf_login = werewolf.mask
            werewolf_uid = self.get_uid_by_login(werewolf_login)
            request.yauser.fields['login'] = werewolf_login
            request.yauser.uid = werewolf_uid

    @staticmethod
    def get_uid_by_login(login):
        from review.staff import logic
        return logic.ensure_person_model(login).uid


class RestrictAccessMiddleware(object):
    def process_request(self, request):
        if settings.ALLOWED_LOGINS:
            allowed_logins = settings.ALLOWED_LOGINS.split(',')
        else:
            allowed_logins = []

        if settings.ALLOWED_SLUGS:
            allowed_slugs = settings.ALLOWED_SLUGS.split(',')
        else:
            allowed_slugs = []

        if settings.ALLOWED_PATHS:
            allowed_paths = settings.ALLOWED_PATHS.split(',')
        else:
            allowed_paths = []

        if not allowed_slugs and not allowed_logins and not allowed_paths:
            return

        import fnmatch
        from review.staff import models
        from review.staff import logic
        login = request.auth.login

        for allowed_path in allowed_paths:
            if fnmatch.fnmatchcase(request.path_info, allowed_path):
                return

        if login in allowed_logins:
            return

        if allowed_slugs:
            allowed_departments = models.Department.objects.filter(
                slug__in=allowed_slugs,
            )
            is_from_allowed_dep = logic.get_persons_from_department_forest(
                roots=list(allowed_departments)
            ).filter(login=login).exists()
            if is_from_allowed_dep:
                return

        error = errors.PermissionDenied(
            "production allowed only for logins {allowed_logins}, "
            "slugs: {allowed_slugs}",
            allowed_logins=allowed_logins,
            allowed_slugs=allowed_slugs,
        )
        return http.JsonResponse(
            status=403,
            data={'errors': {'*': error.as_dict()}},
        )


def serialize_auth(obj):
    if obj is None:
        return
    SIMPLE_TYPES = str, int, dict, type(None)
    SENSITIVE_DATA = (
        'session_id',
        'oauth_token',
    )
    result = {}
    for key, value in obj.__dict__.items():
        if not isinstance(value, SIMPLE_TYPES):
            value = value.__class__.__name__
        if key in SENSITIVE_DATA:
            value = bool(value)
        result[key] = value
    return result
