# coding: utf-8

from django.core.exceptions import ValidationError

from review.lib import strings


def cls_as_code(obj):
    return strings.camelcase_to_underscore(
        type(obj).__name__
    ).upper()


class Error(Exception):

    msg_tpl = ''
    default_msg_params = {}

    def __init__(self, msg_tpl='', **params):
        self.msg_tpl = msg_tpl or self.msg_tpl
        self.code = params.pop('code', cls_as_code(self))
        self.params = params

    @property
    def msg(self):
        msg_params = self.default_msg_params.copy()
        msg_params.update(self.params)
        return self.msg_tpl.format(**msg_params),

    def as_dict(self):
        return {
            'code': self.code,
            'msg': self.msg,
            'params': self.params,
        }


class PermissionDenied(Error):

    msg_tpl = 'User has no {action} permission for {type} {id}'
    default_msg_params = {
        'action': 'read',
        'type': '',
        'id': '',
    }


class NotFound(Error):

    msg_tpl = 'Object {type} {id} was not found'
    default_msg_params = {
        'type': '',
        'id': '',
    }


class FileIncorrect(ValidationError):

    def __init__(self, message='File has incorrect format', code=None, params=None):
        super(FileIncorrect, self).__init__(message, code=code, params=params)
        self.text_code = cls_as_code(self)

    def as_dict(self):
        return {
            'code': self.text_code,
            'msg': self.message,
            'params': self.params,
        }


class CantParseFile(ValidationError):

    def __init__(self, message="Can't parse file", code=None, params=None):
        super(CantParseFile, self).__init__(message, code=code, params=params)
        self.text_code = cls_as_code(self)

    def as_dict(self):
        return {
            'code': self.text_code,
            'msg': self.message,
            'params': self.params,
        }
