# coding: utf-8

from datetime import datetime
import functools
import logging
import os

from django.utils.encoding import force_text


log = logging.getLogger(__name__)
_pid = None


def get_pid():
    global _pid
    if _pid is None:
        _pid = os.getpid()
    return _pid


def timeit(func):

    @functools.wraps(func)
    def deco(*args, **kwargs):
        pieces = _get_pieces(func, *args, **kwargs)
        log.info(
            'starting func %s %s(%s) PID %s',
            pieces['mark_id'],
            pieces['name'],
            pieces['args_str'],
            get_pid(),
        )

        before = datetime.now()
        msg = '%d.%06d TIMEIT for %s %s(%s) PID %s'
        try:
            return func(*args, **kwargs)
        except Exception as exc:
            msg += ' with exception %s' % exc.__class__.__name__
            raise
        finally:
            passed = datetime.now() - before
            log.info(
                msg,
                passed.seconds,
                passed.microseconds,
                pieces['mark_id'],
                pieces['name'],
                pieces['args_str'],
                get_pid(),
            )
    return deco


def timeit_no_args_logging(func):

    @functools.wraps(func)
    def deco(*args, **kwargs):
        pieces = _get_pieces(func)
        log.info(
            'starting func %s %s PID %s',
            pieces['mark_id'],
            pieces['name'],
            get_pid(),
        )

        before = datetime.now()
        msg = '%d.%06d TIMEIT for %s %s PID %s'
        try:
            return func(*args, **kwargs)
        except Exception as exc:
            msg += ' with exception %s' % exc.__class__.__name__
            raise
        finally:
            passed = datetime.now() - before
            log.info(
                msg,
                passed.seconds,
                passed.microseconds,
                pieces['mark_id'],
                pieces['name'],
                get_pid(),
            )
    return deco


def _get_pieces(func, *args, **kwargs):
    args_str = ''
    if args:
        args_str += ', '.join(map(_prepare_for_log, args))
    if kwargs:
        if args_str:
            args_str += ', '
        args_str += ', '.join([
            '%s=%s' % (key, _prepare_for_log(val))
            for key, val in kwargs.items()
        ])
    return {
        'mark_id': func.__module__,
        'name': func.__name__,
        'args_str': args_str,
    }


def _prepare_for_log(obj):
    from django.db import models

    if isinstance(obj, models.Model):
        obj_str = '{}:{}'.format(
            obj.__class__.__name__,
            obj.pk,
        )
    else:
        obj_str = force_text(obj)
    if len(obj_str) > 100:
        obj_str = obj_str[:97] + '...'
    return obj_str
