# coding: utf-8
import logging
import re

from django.utils.cache import patch_vary_headers
from django.utils.deprecation import MiddlewareMixin
from django import http

import yenv
from ylog import context
from corsheaders import middleware as cors_middleware

from review.lib import helpers, auth


logger = logging.getLogger(__name__)
re_accepts_gzip = re.compile(r'\bgzip\b')


class ConfigurableGZipMiddleware(MiddlewareMixin):
    """
    This middleware compresses content if the browser allows gzip compression.
    It sets the Vary header accordingly, so that caches will base their storage
    on the Accept-Encoding header.
    """
    def process_response(self, request, response):
        # It's not worth attempting to compress really short responses.
        if not response.streaming and len(response.content) < 200:
            return response

        # Avoid gzipping if we've already got a content-encoding.
        if response.has_header('Content-Encoding'):
            return response

        patch_vary_headers(response, ('Accept-Encoding',))

        ae = request.META.get('HTTP_ACCEPT_ENCODING', '')
        if not re_accepts_gzip.search(ae):
            return response

        # Return the compressed content only if it's actually shorter.
        compressed_content = compress_string(
            response.content,
            compresslevel=int(request.GET.get('gzip_level', '6'))
        )

        if len(compressed_content) >= len(response.content):
            return response
        response.content = compressed_content
        response['Content-Length'] = str(len(response.content))

        # If there is a strong ETag, make it weak to fulfill the requirements
        # of RFC 7232 section-2.1 while also allowing conditional request
        # matches on ETags.
        etag = response.get('ETag')
        if etag and etag.startswith('"'):
            response['ETag'] = 'W/' + etag
        response['Content-Encoding'] = 'gzip'

        return response


@helpers.timeit
def compress_string(s, compresslevel=None):
    compresslevel = compresslevel or 6
    from io import BytesIO
    from gzip import GzipFile

    zbuf = BytesIO()
    with GzipFile(mode='wb', compresslevel=compresslevel, fileobj=zbuf, mtime=0) as zfile:
        zfile.write(s)
    return zbuf.getvalue()
