# coding: utf-8
from datetime import datetime, timedelta

import boto3
import botocore
import pytz
from django.conf import settings

from tvm2 import TVM2
from ticket_parser2_py3.api.v1 import BlackboxClientId


class TVM2Error(Exception):
    pass


def get_tvm_client():
    return TVM2(
        client_id=settings.TVM_APP_SELF_ID,
        secret=settings.REVIEW_TVM_SECRET,
        blackbox_client=BlackboxClientId.ProdYateam,
        destinations=(i for i in list(settings.TVM_APP_IDS.values()) if i),
    )


def get_service_ticket(client_id):
    """
    :param client_id: str
    :return: str
    """
    tvm_client = get_tvm_client()
    service_tickets = tvm_client.get_service_tickets(client_id)
    service_ticket = service_tickets.get(client_id)
    if not service_ticket:
        raise TVM2Error('Failed to get TVM service ticket for client_id `%s`' % client_id)
    return service_ticket


class Boto3APIError(Exception):
    pass


def get_tvm2_boto3_session(self_client_id, dst, **kwargs):
    """
    from https://a.yandex-team.ru/arc/trunk/arcadia/library/python/awssdk-extensions/src/session.py?rev=6525377
    """
    # access_key is only required for SQS
    access_key = 'TVM_V2_{}'.format(self_client_id)

    def refresh():
        ticket = get_service_ticket(client_id=dst)
        expiry_time = (datetime.now(pytz.utc) + timedelta(hours=1)).isoformat()
        return dict(
            access_key=access_key,
            secret_key='unused',
            token='TVM2 {}'.format(ticket),
            expiry_time=expiry_time,
        )

    credentials = botocore.credentials.RefreshableCredentials.create_from_metadata(
        metadata=refresh(),
        refresh_using=refresh,
        method='tvm2',
    )
    session = botocore.session.get_session()
    # '_credentials' is not public API so we cannot fully rely on it
    # Need to check and alert early
    if not hasattr(session, '_credentials'):
        raise Boto3APIError("botocore API has changed, session no longer has '_credentials' attribute")
    session._credentials = credentials
    return boto3.Session(botocore_session=session, **kwargs)
