# coding: utf-8
import json
import logging

from ylog import context as log_context

from django.views import View as DjangoView
from django import http

from review.lib import (
    auth as auth_module,
    encryption,
    errors,
    helpers,
    serializers,
    file_properties,
)


log = logging.getLogger(__name__)


class FileResponse(http.HttpResponse):
    def __init__(self, **kwargs):
        fp = kwargs.pop('file_properties')
        kwargs['content_type'] = fp.content_type
        super(FileResponse, self).__init__(**kwargs)
        self['Content-Disposition'] = 'attachment; filename=%s.%s' % (fp.file_name, fp.extension)


class View(DjangoView):
    """
    Базовая вьюха для проекта, преследует такие цели:
      1) не хочется писать JsonResponse(...) в каждой view, а
         99.99% возвращают json
      2) не писать запихивание входных данных в формы и обработку ошибок в
         каждой view, почти всегда это нужно.
    Однако каждая фича должна оставаться опциональной и должно быть можно
    переопределить get и post и прочие методы и все должно работать
    как обычно.
    """

    # можно выключить заворачивание всего в JsonResponse
    response_format = 'json'
    # по умолчанию сериализуем без параметров
    json_response_params_default = {}
    # можно переопределять в своей вьюхе
    json_response_params = {}
    include_auth_to_response = True

    form_cls = None
    form_cls_get = None
    form_cls_post = None

    exception_to_status_code = {
        errors.Error: 400,
        errors.PermissionDenied: 403,
        errors.NotFound: 404,
    }
    exceptions_to_handle = tuple(exception_to_status_code)

    log_params_for = set()
    WHITE_LIST_TO_LOG = set()

    def dispatch(self, request, *args, **kwargs):
        log.debug('{}: {} {} {}'.format(
            request.method,
            request.path,
            self.get_query_dict(request),
            kwargs
        ))
        method_lower = request.method.lower()
        handler = getattr(self, method_lower, None)
        specific_process_handler_exists = hasattr(self, 'process_' + method_lower)
        response_format = kwargs.get('format', self.response_format)
        if handler is not None:
            response = handler(request, *args, **kwargs)
        elif specific_process_handler_exists:
            try:
                response = self.dispatch_process(request, *args, **kwargs)
            except self.exceptions_to_handle as exc:
                response = self.handle_exception(exc, request, args, kwargs)
                if response is None:
                    raise
        else:
            response = self.http_method_not_allowed(request, *args, **kwargs)
        return self.do_response(
            request,
            response,
            response_format=response_format,
        )

    def dispatch_process(self, request, *args, **kwargs):
        method_lower = request.method.lower()
        log.info('start processing view for {} {} PID {}'.format(
            method_lower,
            request.path,
            helpers.get_pid(),
        ))
        form = self.maybe_get_form(request)
        process_handler = getattr(self, 'process_' + method_lower)
        if form is not None:
            if form.is_valid():
                data = form.cleaned_data
                data.update(kwargs)
                with log_context.LogContext(**self.get_logging_context(data)):
                    self.log_params(method_lower, data)
                    return process_handler(
                        auth=request.auth,
                        data=data,
                    )
            else:
                error_response = self.get_errors_from_invalid_form(form)
                return self.do_response(
                    request,
                    {'errors': error_response},
                    response_format='json',
                    status_code=400,
                )
        else:
            data = self.get_query_dict(request)
            data.update(kwargs)
            with log_context.LogContext(**self.get_logging_context(data)):
                self.log_params(method_lower, data)
                return process_handler(
                    auth=request.auth,
                    data=data
                )

    def get_errors_from_invalid_form(self, form):
        result = {}
        for key, value in form.errors.items():
            if len(value) == 1:
                if hasattr(value, 'data'):
                    err_obj = value.data[0]
                    params = err_obj.params or {}
                    code = errors.cls_as_code(err_obj)
                else:
                    params = {}
                    code = 'VALIDATION_ERROR'
                result[key] = {
                    'code': code,
                    'msg': value[0],
                    'params': params,
                }
            else:
                for index, item in enumerate(value):
                    result[key + '_' + str(index)] = {
                        'code': 'VALIDATION_ERROR',
                        'msg': value[0],
                        'params': {},
                    }
        return result

    def do_response(self, request, response, response_format='json', status_code=200):
        if not isinstance(response, (dict, list)):
            return response

        assert response_format in ('json', 'csv', 'xls'), 'Only "json", "csv", "xls" supported'
        serialized_data = self.serialize_data(data=response, response_format=response_format)

        if response_format == 'json':
            return self.make_json_response(request=request, data=serialized_data, status=status_code)

        return self.make_file_response(
            request=request, data=serialized_data, status=status_code, response_format=response_format
        )

    def serialize_data(self, data, response_format):
        if response_format == 'json':
            return data

        assert isinstance(data, list), 'Only list can serialized to file formats'

        if response_format == 'xls':
            return serializers.XLSSerializer.serialize(
                data=data, file_properties=self.get_file_properties(response_format)
            )
        else:
            return serializers.CSVSerializer.serialize(data)

    def make_json_response(self, request, data, status):
        if isinstance(data, dict):
            params = dict(self.json_response_params_default)
            params.update(self.json_response_params)
            if self.include_auth_to_response:
                data['_auth'] = auth_module.serialize_auth(request.auth)
        else:
            params = dict(safe=False)

        return http.JsonResponse(data=data, status=status, **params)

    def make_file_response(self, request, data, status, response_format):
        return FileResponse(
            content=data,
            status=status,
            file_properties=self.get_file_properties(response_format),
        )

    def get_file_properties(self, response_format):
        if response_format == 'csv':
            return file_properties.CsvFileProperties()
        else:
            return file_properties.XlsFileProperties()

    def handle_exception(self, exc, request, args, kwargs):
        log.info('Exception: {}'.format(exc))
        response_dict = {
            'errors': {
                '*': exc.as_dict(),
            }
        }
        status_code = self.exception_to_status_code.get(exc.__class__, 400)
        return self.do_response(
            request=request,
            response=response_dict,
            status_code=status_code,
        )

    def get_query_dict(self, request):
        if request.method == 'GET':
            return request.GET.copy()
        else:
            if request.content_type is None or request.content_type == 'application/json':
                query_dict = self.get_json_from_request(request)
            else:
                query_dict = request.POST.copy()
            query_dict.update(request.GET.copy())
        return query_dict

    def get_files(self, request):
        return request.FILES

    def get_from_cls(self, request):
        specific_form_cls_name = 'form_cls_' + request.method.lower()
        return getattr(self, specific_form_cls_name) or self.form_cls

    def maybe_get_form(self, request):
        form_cls = self.get_from_cls(request)
        if form_cls is None:
            return
        return form_cls(
            data=self.get_query_dict(request),
            files=self.get_files(request),
        )

    def get_json_from_request(self, request):
        if not hasattr(request, '_json'):
            try:
                request._json = json.load(request)
            except ValueError:
                log.exception('Failed deserialize json')
                request._json = {}
        return request._json

    def get_logging_context(self, data):
        return {}

    def log_params(self, method, data):
        if method not in self.log_params_for:
            return
        to_log = {}
        for k, v in data.items():
            if k not in self.WHITE_LIST_TO_LOG:
                try:
                    v = str(v)
                    v = encryption.encrypt(v)
                except ValueError:
                    log.warning('Can\t encrypt value len of %s', len(v))
                    v = 'Failed to crypt'
            to_log[k] = v
        log.info('View %s called with params: %s', self.__class__.__name__, to_log)
