# coding: utf-8

import json

from django.contrib import admin
from django.conf import settings
from post_office import models as post_office_models
from post_office import admin as post_office_admin

from review.notifications import models as notification_models


admin.site.unregister(post_office_models.Email)
admin.site.unregister(post_office_models.EmailTemplate)
admin.site.unregister(post_office_models.Attachment)


@admin.register(post_office_models.Email)
class EmailAdmin(post_office_admin.EmailAdmin):

    list_display = (
        'id',
        'to',
        'receiver_login',
        'status',
        'created',
        'subject',
        'retries_left',
    )

    search_fields = (
        'context',
    )

    _exclude_fields = {
        'message',
        'html_message',
        'context',
        'template',
        'priority',
    }
    extra_fields = [
        'receiver_login'
    ]

    if not settings.IS_PRODUCTION_DB:
        extra_fields += [
            'html_message_rendered',
            'context_prettyprinted',
        ]

    fields = []

    for field in post_office_models.Email._meta.fields:
        if field.name not in _exclude_fields:
            fields.append(field.name)

    fields.extend(extra_fields)

    readonly_fields = fields

    def receiver_login(self, obj):
        return obj.context.get('receiver_login')

    def retries_left(self, obj):
        if hasattr(obj, 'retry_counter'):
            return obj.retry_counter.counter
        else:
            return notification_models.RETRIES_NUMBER

    def context_prettyprinted(self, obj):
        return '<pre>' + json.dumps(obj.context, indent=2) + '</pre>'
    context_prettyprinted.allow_tags = True

    def html_message_rendered(self, obj):
        return obj.html_message
    html_message_rendered.allow_tags = True

    def get_queryset(self, request):
        return super(EmailAdmin, self).get_queryset(request).select_related(
            'retry_counter'
        )
