# coding: utf-8

import pytz
import functools
import logging
import datetime
from collections import defaultdict
from itertools import chain

from django.db.models import Q
from django.conf import settings

from review.notifications import const as notifications_const
from review.notifications import helpers
from review.notifications import notifications
from review.staff.models import Person
from review.core.models import PersonReviewChange
from review.core import const as core_const
from review.core.logic.helpers import (
    get_chief_ids,
    get_first_reviewer_ids,
    get_persons_details,
    get_reviews_details,
    localized,
)

log = logging.getLogger(__name__)


DIGEST_PERIOD = 1  # days
TYPES = notifications_const.TYPES
FIELDS = core_const.FIELDS
PERSON_REVIEW_STATUS = core_const.PERSON_REVIEW_STATUS


def is_disabled(notification_type):
    return notification_type in settings.DISABLE_NOTIFICATION_TYPES


def build_notifications_for_person_review_changes(changes):
    grouped_by_type = {}
    for change in changes:
        diff = change.diff

        status_change = diff.get(core_const.FIELDS.STATUS, {})
        new_status = status_change.get('new')
        old_status = status_change.get('old')
        approve_change = diff.get(core_const.FIELDS.APPROVE_LEVEL, dict(new=0, old=0))

        is_approved = (
            approve_change['new'] > approve_change['old'] or
            new_status == PERSON_REVIEW_STATUS.APPROVED
        )
        is_unapproved = (
            approve_change['new'] < approve_change['old'] or

            new_status == PERSON_REVIEW_STATUS.APPROVAL and
            old_status == PERSON_REVIEW_STATUS.APPROVED
        )
        if is_approved:
            notification_type = TYPES.APPROVED
        elif is_unapproved:
            notification_type = TYPES.UNAPPROVED
        elif any(flag_field in diff for flag_field in core_const.FIELDS.FLAG_FIELDS):
            notification_type = TYPES.FLAGGED
        elif new_status == PERSON_REVIEW_STATUS.ANNOUNCED:
            notification_type = TYPES.ANNOUNCED
        else:
            notification_type = TYPES.NO_HANDLE
        grouped_by_type.setdefault(notification_type, []).append(change)

    created_notifications = []
    success_ids = []
    type_to_bulk_builder = {
        TYPES.FLAGGED: _build_notifications_for_flag,
        TYPES.APPROVED: _build_notifications_for_approve,
        TYPES.UNAPPROVED: _build_notifications_for_unapprove,
    }
    for notification_type, group_changes in grouped_by_type.items():
        if is_disabled(notification_type):
            success_ids += (ch.id for ch in group_changes)
            continue
        if notification_type in type_to_bulk_builder:
            builder = type_to_bulk_builder[notification_type]
            with helpers.no_fail_on_exception():
                created_notifications += builder(group_changes)
                success_ids += (ch.id for ch in group_changes)
        elif notification_type == TYPES.ANNOUNCED:
            need_to_notify = (
                ch for ch in group_changes
                if ch.person_review.review.notify_events_other
            )
            created_notifications += list(map(_build_notifications_for_announce,
                                              need_to_notify))
            success_ids += (ch.id for ch in group_changes)
        elif notification_type == TYPES.NO_HANDLE:
            success_ids += (ch.id for ch in group_changes)
    return dict(
        notifications=created_notifications,
        success_ids=success_ids,
    )


def _build_notifications_for_announce(change):
    person_review = change.person_review
    person = person_review.person
    return notifications.PersonReviewAnnounceNotification(
        receiver=person,
        context={
            'person_review': helpers.srlz_person_review(
                person_review,
                language=person.language,
            ),
            'review': helpers.srlz_review(
                person_review.review,
                language=person.language,
            ),
        },
    )


def _build_notifications_for_flag(changes):
    by_receiver = helpers.group_changes_by_receiver_and_review_and_change_author(changes)

    result = []
    for receiver_group in list(by_receiver.values()):
        receiver = receiver_group['receiver']
        for review_data in list(receiver_group['reviews'].values()):
            context = {}
            review_serialized = helpers.srlz_review(
                obj=review_data['review'],
                language=receiver.language,
            )
            context['review'] = review_serialized
            change_data_serialized = context['change_data'] = []
            for change_data in list(review_data['change_data'].values()):
                change_datum = {
                    'change_author': helpers.srlz_person(
                        obj=change_data['change_author'],
                        language=receiver.language,
                    ),
                    'changes': {
                        'flagged': [],
                        'unflagged': []
                    }
                }
                for change in change_data['changes']:
                    diff = change.diff.get(
                        core_const.FIELDS.FLAGGED,
                        change.diff.get(core_const.FIELDS.FLAGGED_POSITIVE),
                    )
                    bucket = 'flagged' if diff['new'] else 'unflagged'
                    change_datum['changes'][bucket].append(
                        {
                            'person_review': helpers.srlz_person_review(
                                change.person_review,
                                language=receiver.language,
                            )
                        }
                    )
                change_data_serialized.append(change_datum)

            result.append(
                notifications.PersonReviewFlagNotification(
                    receiver=receiver,
                    context=context,
                )
            )

    return result


def _build_notifications_for_approve_change(notification_cls, changes):
    by_receiver = helpers.group_changes_by_receiver_and_review_and_change_author(changes)

    result = []
    for receiver_group in list(by_receiver.values()):
        receiver = receiver_group['receiver']
        for review_data in list(receiver_group['reviews'].values()):
            context = {}
            review_serialized = helpers.srlz_review(
                obj=review_data['review'],
                language=receiver.language,
            )
            context['review'] = review_serialized
            change_data_serialized = context['change_data'] = []
            for change_data in list(review_data['change_data'].values()):
                change_datum = {
                    'change_author': helpers.srlz_person(
                        obj=change_data['change_author'],
                        language=receiver.language,
                    ),
                    'changes': []
                }
                for change in change_data['changes']:
                    change_datum['changes'].append(
                        {
                            'person_review': helpers.srlz_person_review(
                                change.person_review,
                                language=receiver.language,
                            )
                        }
                    )
                change_data_serialized.append(change_datum)

            result.append(
                notification_cls(
                    receiver=receiver,
                    context=context,
                )
            )

    return result


_build_notifications_for_approve = functools.partial(
    _build_notifications_for_approve_change,
    notifications.PersonReviewApproveNotification,
)

_build_notifications_for_unapprove = functools.partial(
    _build_notifications_for_approve_change,
    notifications.PersonReviewUnapproveNotification,
)


def build_notifications_for_person_review_comments(comments):
    by_receiver = {}
    """
    expected structure
    {
        receiver_id: {
            'receiver': receiver_obj,
            'reviews': {
                review_id: {
                    'review': review_obj,
                    'comment_data': {
                        person_review_id: {
                            'person_review': person_review_obj,
                            'comments': [comment_objs]
                        }
                    }
                }
            }
        }
    }
    """
    result = {
        'notifications': [],
        'success_ids': [comment.id for comment in comments],
    }
    if is_disabled(TYPES.COMMENTED):
        return result

    for comment in comments:
        receivers = helpers.get_person_review_comment_receivers(comment)
        for receiver in receivers:
            receiver_group = by_receiver.setdefault(receiver.id, {
                'receiver': receiver,
                'reviews': {}
            })
            by_review = receiver_group['reviews']
            review_group = by_review.setdefault(comment.person_review.review_id, {
                'review': comment.person_review.review,
                'comment_data': {},
            })
            by_person_review = review_group['comment_data']
            change_author_group = by_person_review.setdefault(
                comment.person_review_id, {
                    'person_review': comment.person_review,
                    'comments': []
                }
            )
            change_author_group['comments'].append(comment)

    for receiver_group in list(by_receiver.values()):
        receiver = receiver_group['receiver']
        for review_data in list(receiver_group['reviews'].values()):
            context = {}
            review_serialized = helpers.srlz_review(
                obj=review_data['review'],
                language=receiver.language,
            )
            context['review'] = review_serialized
            comment_data_serialized = context['comment_data'] = []
            for comment_data in list(review_data['comment_data'].values()):
                comment_datum = {
                    'person_review': helpers.srlz_person_review(
                        obj=comment_data['person_review'],
                        language=receiver.language,
                    ),
                    'comments': []
                }
                for comment in comment_data['comments']:
                    receiver_tz = pytz.timezone(receiver.timezone)
                    updated_at = comment.updated_at or comment.created_at
                    updated_at = updated_at.astimezone(receiver_tz)
                    updated_at_str = updated_at.strftime(settings.MAIN_DATETIME_FORMAT)
                    comment_datum['comments'].append(
                        {
                            'subject': helpers.srlz_person(
                                obj=comment.subject,
                                language=receiver.language,
                            ),
                            'text_wiki': comment.text_wiki,
                            'updated_at': updated_at_str,
                        }
                    )
                comment_data_serialized.append(comment_datum)

            result['notifications'].append(
                notifications.PersonReviewCommentNotification(
                    receiver=receiver,
                    context=context,
                )
            )
    return result


def build_notifications_for_review_reminders(fetched):
    result = {'notifications': []}
    notifications_list = result['notifications']

    if is_disabled(TYPES.DATES_REMINDER):
        return result

    person_reviews = fetched
    by_receiver = {}
    for person_review in person_reviews:
        receivers = helpers.get_person_review_action_at_receivers(person_review)
        for receiver in receivers:
            receiver_data = by_receiver.setdefault(receiver.id, {
                'receiver': receiver,
                'reviews': {}
            })
            by_review = receiver_data['reviews']
            review = person_review.review
            review_data = by_review.setdefault(review.id, {
                'review': review,
                'person_reviews': []
            })
            review_data['person_reviews'].append(person_review)

    for receiver_data in list(by_receiver.values()):
        receiver = receiver_data['receiver']
        for review_data in list(receiver_data['reviews'].values()):
            review = review_data['review']
            person_reviews = review_data['person_reviews']
            wait_evaluation_count = len([
                pr for pr in person_reviews
                if pr.status == core_const.PERSON_REVIEW_STATUS.WAIT_EVALUATION
            ])
            wait_approval_count = len([
                pr for pr in person_reviews
                if pr.status in (
                    core_const.PERSON_REVIEW_STATUS.EVALUATION,
                    core_const.PERSON_REVIEW_STATUS.APPROVAL,
                )
            ])
            if wait_evaluation_count + wait_approval_count == 0:
                continue

            notifications_list.append(
                notifications.ReviewReminderNotification(
                    receiver=receiver,
                    context={
                        'review': helpers.srlz_review(
                            review,
                            language=receiver.language,
                            _extra_fields={
                                'finish_submission_date',
                                'finish_approval_date',
                            }
                        ),
                        'language': receiver.language,
                        'wait_evaluation_count': wait_evaluation_count,
                        'wait_approval_count': wait_approval_count,
                    }
                )
            )

    return result


def build_notifications_for_added_calibrators(fetched):
    calibration_to_admins = defaultdict(list)
    for role in fetched['admin_roles']:
        calibration_to_admins[role.calibration_id].append(role.person)

    result = []
    success_ids = []
    for role in fetched['calibrator_roles']:
        success_ids.append(role.id)
        if not role.calibration.notify_users:
            continue
        if is_disabled(TYPES.CALIBRATORS_ADDED):
            continue
        calibrator = role.person
        language = calibrator.language
        admins = [
            helpers.srlz_person(obj=admin, language=language)
            for admin in calibration_to_admins[role.calibration_id]
        ]
        context = dict(
            calibration=helpers.srlz_calibration(role.calibration),
            admins=admins,
            language=language,
        )
        result.append(notifications.CalibratorAddedNotification(
            receiver=calibrator,
            context=context,
        ))

    return {
        'notifications': result,
        'success_ids': success_ids,
    }


def build_notifications_for_personreview_changes():
    """
    Выбираем PersonReviewChange за последние сутки те, которые про смену грейда либо оценки.
    Возвращаем данные по автору изменений оценки либо грейда по всем изменённым person_review,
        сгруппировав их по получателям (первым ревьюверам и непосредственным руководителям).
    """
    now = datetime.datetime.now()
    yesterday = now - datetime.timedelta(days=DIGEST_PERIOD)
    level_changed = Q(diff__contains='level_change')
    mark_changed = Q(diff__contains='"mark"')

    changes_data = list(
        PersonReviewChange.objects
        .filter(created_at__gte=yesterday, created_at__lte=now)
        .filter(level_changed | mark_changed)
        .exclude(subject_type=core_const.PERSON_REVIEW_CHANGE_TYPE.ROBOT)
        .values_list('person_review_id', 'diff', 'person_review__person_id', 'subject_id')
    )

    person_review_ids, person_ids, author_ids = [], [], []
    for pr_id, _, person_id, author_id in changes_data:
        person_review_ids.append(pr_id)
        person_ids.append(person_id)
        author_ids.append(author_id)

    pr_details = get_reviews_details(person_review_ids)  # {int: {str: int}, }
    reviewer_ids = get_first_reviewer_ids(person_review_ids)  # {int: [int, int, ], }
    chief_ids = get_chief_ids(person_ids=person_ids)  # {int: int}
    all_person_ids = (
        set(person_ids) |
        set(author_ids) |
        set(chief_ids.values()) |
        set(chain.from_iterable(list(reviewer_ids.values())))
    )
    persons_details = get_persons_details(all_person_ids)

    # сгруппированный по получателям dict
    context = defaultdict(list)

    for pr_id, diff, person_id, author_id in changes_data:
        reciever_ids = set(reviewer_ids[pr_id])
        if person_id in chief_ids:
            reciever_ids |= {chief_ids[person_id]}
        for reciever_id in reciever_ids - {author_id}:
            reciever_language = persons_details[reciever_id]['language']
            reciever_email = persons_details[reciever_id]['work_email']
            context[reciever_email].append(
                {
                    'mark_changed_by': localized(
                        persons_details[author_id],
                        reciever_language
                    ) if 'mark' in diff else {},
                    'level_changed_by': localized(
                        persons_details[author_id],
                        reciever_language
                    ) if 'level_change' in diff else {},
                    'person': localized(persons_details[person_id], reciever_language),
                    'person_review_id': pr_id,
                    'review_id': pr_details[pr_id]['review_id'],
                    'review_name': pr_details[pr_id]['review__name'],
                }
            )
    result = []
    person_instances = {
        person.work_email: person
        for person in Person.objects.filter(work_email__in=list(context.keys()))
    }
    for reciever_email, email_context in context.items():
        result.append(notifications.PersonReviewChangedDigestNotification(
            receiver=person_instances[reciever_email],
            context={'change_data': email_context},
        ))

    return {'notifications': result}
