# coding: utf-8

from django.conf import settings
import arrow
from django.db.models import Q

from review.lib import wf
from review.core import const as core_const
from review.core import models as core_models


def fetch_changes_for_period(minutes=None):
    return _fetch_models_for_period(
        model_cls=core_models.PersonReviewChange,
        minutes=minutes,
    )


def fetch_comments_for_period(minutes=None):
    comments = _fetch_models_for_period(
        model_cls=core_models.PersonReviewComment,
        minutes=minutes,
    )[:settings.MAX_OBJECTS_PER_TASK]
    formatted = format_comments(it.text_wiki for it in comments)
    for comment in comments:
        comment.text_wiki = formatted[comment.text_wiki]
    return comments


def format_comments(texts):
    return wf.format_markup_bulk(texts)


def _fetch_models_for_period(model_cls, minutes=None):
    query_filter = Q(notified=False)
    if minutes is not None:
        before_time = arrow.now().replace(
            minutes=-minutes
        ).datetime
        query_filter &= Q(created_at__gte=before_time)

    models = model_cls.objects.filter(
        query_filter,
    ).select_related(
        'subject',
        'person_review',
    ).prefetch_related(
        'person_review__review',
        'person_review__review__roles',
        'person_review__roles',
        'person_review__roles__person',
    )
    return models


def fetch_data_for_reviews_reminders(review_ids=None):
    today = arrow.now().date()
    week_day = core_const.WEEK_DAYS[today.weekday()]

    review_filter_q = Q(
        review__notify_reminder_date_to__gte=today,
        review__notify_reminder_date_from__lte=today,
        review__notify_reminder_days__contains=week_day,
    ) | Q(
        review__notify_reminder_date_from=today,
        review__notify_reminder_date_to__isnull=True,
    )

    if review_ids:
        review_filter_q &= Q(review_id__in=review_ids)

    person_reviews = core_models.PersonReview.objects.filter(
        review_filter_q,
        review__start_date__lte=today,
        review__status=core_const.REVIEW_STATUS.IN_PROGRESS,
        status__in=(
            core_const.PERSON_REVIEW_STATUS.WAIT_EVALUATION,
            core_const.PERSON_REVIEW_STATUS.EVALUATION,
            core_const.PERSON_REVIEW_STATUS.APPROVAL,
        ),
    ).prefetch_related(
        'review',
        'roles',
        'roles__person',
    )
    return person_reviews


def fetch_data_for_added_calibrators():
    calibrators_to_notify = core_models.CalibrationRole.objects.filter(
        type=core_const.ROLE.CALIBRATION.CALIBRATOR,
        notified=False,
        calibration__status=core_const.CALIBRATION_STATUS.IN_PROGRESS,
    ).select_related('person')
    calibration_ids = {it.calibration_id for it in calibrators_to_notify}
    admin_roles = core_models.CalibrationRole.objects.filter(
        type=core_const.ROLE.CALIBRATION.ADMIN,
        calibration_id__in=calibration_ids,
    ).select_related('person')
    return dict(
        calibrator_roles=calibrators_to_notify,
        admin_roles=admin_roles,
    )
