# coding: utf-8

import logging
import contextlib

from review.core import const as core_const
from review.core.logic.assemble import setters_person_review
from review.core.logic.assemble import fetch_person_reviews
from review.notifications import serializers


log = logging.getLogger(__name__)


def srlz_review(obj, **context):
    serializer = serializers.ReviewSerializer
    extra_fields = set(context.pop('_extra_fields', []))
    return serializer.serialize(
        obj=obj,
        context=context,
        fields_requested=serializer.default_fields | set(extra_fields)
    )


def srlz_person(obj, **context):
    return serializers.PersonSerializer.serialize(
        obj=obj,
        context=context,
    )


def srlz_person_review(obj, **context):
    return serializers.PersonReviewSerializer.serialize(
        obj=obj,
        context=context,
    )


def srlz_calibration(obj, **context):
    return serializers.CalibrationSerializer.serialize(
        obj=obj,
        context=context,
    )


def group_changes_by_receiver_and_review_and_change_author(changes):
    by_receiver = {}
    for change in changes:
        receivers = get_person_review_change_receivers(change)
        for receiver in receivers:
            receiver_group = by_receiver.setdefault(receiver.id, {
                'receiver': receiver,
                'reviews': {}
            })
            by_review = receiver_group['reviews']
            review_group = by_review.setdefault(change.person_review.review_id, {
                'review': change.person_review.review,
                'change_data': {},
            })
            by_change_author = review_group['change_data']
            change_author_group = by_change_author.setdefault(
                change.subject_id, {
                    'change_author': change.subject,
                    'changes': []
                }
            )
            exists_change_with_person_review = any(
                existing.person_review == change.person_review
                for existing in change_author_group['changes']
            )
            if not exists_change_with_person_review:
                change_author_group['changes'].append(change)
    return by_receiver


def get_person_review_change_receivers(change):
    result = set()
    review = change.person_review.review
    if review.notify_events_other:
        if core_const.FIELDS.APPROVE_LEVEL in change.diff:
            # помним, что есть кейс с апрувом с пропуском ревьюеров —
            # их нужно тоже известить — поэтому смотрим на 'new'
            approve_level = change.diff[core_const.FIELDS.APPROVE_LEVEL]['new'] - 1
        else:
            approve_level = change.person_review.approve_level
        result |= _get_reviewers_by_approve_level(
            person_review=change.person_review,
            approve_level=approve_level,
        )
    if review.notify_events_superreviewer:
        result |= _get_superreviewers_persons(review)
    result -= {change.subject}
    return result


def get_person_review_comment_receivers(comment):
    result = set()
    review = comment.person_review.review
    if review.notify_events_other:
        result |= _get_reviewers_by_approve_level(
            person_review=comment.person_review,
            approve_level=comment.person_review.approve_level,
        )
    if review.notify_events_superreviewer:
        result |= _get_superreviewers_persons(review)
    result -= {comment.subject}
    return result


def _get_reviewers_by_approve_level(person_review, approve_level):
    return set(
        role.person for role in person_review.roles.all()
        if role.position <= approve_level and
        role.type in (
            core_const.ROLE.PERSON_REVIEW.REVIEWER,
            core_const.ROLE.PERSON_REVIEW.TOP_REVIEWER,
        )
    )


def _get_superreviewers_persons(review):
    return {
        role.person
        for role in review.roles.all()
        if role.type == core_const.ROLE.REVIEW.SUPERREVIEWER
    }


def get_person_review_action_at_receivers(person_review):
    reviewers = [
        role for role in person_review.roles.all()
        if role.type in (
            core_const.ROLE.PERSON_REVIEW.REVIEWER,
            core_const.ROLE.PERSON_REVIEW.TOP_REVIEWER,
        )
    ]
    reviewers = fetch_person_reviews.regroup_reviewers_helper(
        roles_flat=sorted(reviewers, key=lambda it: it.position),
        _format='objects',
    )[person_review.id]

    action_at_roles = setters_person_review.action_at_helper(
        reviewers=reviewers,
        status=person_review.status,
        approve_level=person_review.approve_level,
    )
    return [role.person for role in action_at_roles]


def mark_notified(model_cls, ids):
    model_cls.objects.filter(id__in=ids).update(notified=True)


@contextlib.contextmanager
def no_fail_on_exception(msg='failed'):
    try:
        yield
    except Exception:
        log.exception(msg)
