# coding: utf-8

import logging
from copy import deepcopy

from django.template import loader
from django.utils import translation
from django.conf import settings
from django.core import mail
import premailer

from review.notifications import helpers


class PersonalNotification(object):

    body_template = None
    _body_template_cache = {}

    def __init__(self, receiver, context):
        self.receiver = receiver
        self.language = self.receiver.language
        self.context = deepcopy(context)

    @classmethod
    def get_body_template(cls, language):
        template_name = cls.body_template + '_' + language + '.html'
        return loader.get_template(template_name)

    def get_extended_context(self):
        context = {
            # пишем subject в title страницы для удобства отладки
            'subject': self.get_subject(),
            'language': self.language,
            'receiver': helpers.srlz_person(
                self.receiver,
                language=self.language,
            ),
            'settings': settings,
        }
        context.update(self.context)
        return context

    def get_debug_context(self):
        context = {
            'language': self.language,
            'receiver_login': self.receiver.login,
        }
        context.update(self.context)
        return context

    def get_body(self):
        body_template = self.get_body_template(language=self.language)
        translation.activate(self.language)
        rendered = body_template.render(self.get_extended_context())
        rendered = premailer.Premailer(
            html=rendered,
            cssutils_logging_level=logging.ERROR,
        ).transform()
        return rendered

    def get_subject(self):
        return getattr(self, 'get_subject_' + self.language)()

    def get_subject_ru(self):
        return 'Без темы'

    def get_subject_en(self):
        return 'No subject'

    def get_to(self):
        return [self.receiver.work_email]

    def get_from(self):
        return settings.DEFAULT_FROM_EMAIL

    def get_cc(self):
        return []

    def get_bcc(self):
        return []

    def get_reply_to(self):
        return [settings.DEFAULT_REPLY_TO_EMAIL]

    def get_headers(self):
        return {}

    def get_message_id_header(self, type, id):
        id_string = "%s.%d" % (type, id)
        return {
            'Message-ID': mail.make_msgid(idstring=id_string),
        }


class ReviewReminderNotification(PersonalNotification):
    body_template = 'review_dates_reminder'

    def get_subject_ru(self):
        review_name = self.context['review']['name']
        return 'Напоминание о ревью ({})'.format(review_name)

    def get_subject_en(self):
        review_name = self.context['review']['name']
        return 'Reminder about the review ({})'.format(review_name)

    def get_headers(self):
        return self.get_message_id_header(
            type='review',
            id=self.context['review']['id'],
        )


class PersonReviewCommentNotification(PersonalNotification):
    body_template = 'person_review_comment'

    def get_subject_ru(self):
        review_name = self.context['review']['name']
        return 'Новый комментарий к ревью ({})'.format(review_name)

    def get_subject_en(self):
        review_name = self.context['review']['name']
        return 'A new comment on the review ({})'.format(review_name)

    def get_headers(self):
        return self.get_message_id_header(
            type='review',
            id=self.context['review']['id'],
        )


class PersonReviewFlagNotification(PersonalNotification):
    body_template = 'person_review_flag'

    def get_subject_ru(self):
        review_name = self.context['review']['name']
        return 'Флаг «Нужно обсудить» изменен ({})'.format(review_name)

    def get_subject_en(self):
        review_name = self.context['review']['name']
        return 'Need discuss flag changed ({})'.format(review_name)

    def get_headers(self):
        return self.get_message_id_header(
            type='review',
            id=self.context['review']['id'],
        )


class PersonReviewApproveNotification(PersonalNotification):
    body_template = 'person_review_approve'

    def get_subject_ru(self):
        review_name = self.context['review']['name']
        return 'Ревью согласовано ({})'.format(review_name)

    def get_subject_en(self):
        review_name = self.context['review']['name']
        return 'Review agreed ({})'.format(review_name)

    def get_headers(self):
        return self.get_message_id_header(
            type='review',
            id=self.context['review']['id'],
        )


class PersonReviewUnapproveNotification(PersonalNotification):
    body_template = 'person_review_unapprove'

    def get_subject_ru(self):
        review_name = self.context['review']['name']
        return 'Отмена согласования ревью ({})'.format(review_name)

    def get_subject_en(self):
        review_name = self.context['review']['name']
        return 'Cancellation of approval ({})'.format(review_name)

    def get_headers(self):
        return self.get_message_id_header(
            type='review',
            id=self.context['review']['id'],
        )


class PersonReviewAnnounceNotification(PersonalNotification):
    body_template = 'person_review_announce'

    def get_subject_ru(self):
        review_name = self.context['person_review']['review']['name']
        return 'Опубликованы результаты ревью ({})'.format(review_name)

    def get_subject_en(self):
        review_name = self.context['person_review']['review']['name']
        return 'Review published ({})'.format(review_name)

    def get_headers(self):
        return self.get_message_id_header(
            type='person_review',
            id=self.context['person_review']['id'],
        )


class CalibratorAddedNotification(PersonalNotification):
    body_template = 'calibrator_added'

    def get_subject_ru(self):
        return 'Калибровка «{}»'.format(self.context['calibration']['name'])

    def get_subject_en(self):
        return 'Calibration "{}"'.format(self.context['calibration']['name'])

    def get_headers(self):
        return self.get_message_id_header(
            type='calibration',
            id=self.context['calibration']['id'],
        )


class PersonReviewChangedDigestNotification(PersonalNotification):
    body_template = 'personreview_update_digest'

    def get_subject_ru(self):
        return "Изменения в ревью ваших сотрудников"

    def get_subject_en(self):
        return "Changes to your employees' reviews"
