import time

from django.db import models, transaction
from django_bulk_update.helper import bulk_update
from post_office import models as post_office_models
from . import models as notification_models


MIN_SECS_SINCE_LAST_TRY = {
    1: 10 * 60,
    2: 3600,
    3: 3 * 3600,
    4: 12 * 3600
}


def retry_failed():
    is_failed = models.Q(status=post_office_models.STATUS.failed)
    has_no_counter = models.Q(retry_counter__isnull=True)
    counter_gte_zero = models.Q(retry_counter__counter__gt=0)
    fail_query = is_failed & (has_no_counter | counter_gte_zero)
    emails = post_office_models.Email.objects.filter(
        fail_query
    ).select_related('retry_counter')
    create, update = [], []
    emails_to_update = []
    for email in emails:
        if not hasattr(email, 'retry_counter'):
            create.append(notification_models.EmailRetryCount(email=email))
            num_of_try = 1
        elif email.retry_counter.counter > 0:
            counter = email.retry_counter
            num_of_try = notification_models.RETRIES_NUMBER - counter.counter
            counter.counter -= 1
            update.append(counter)
        else:
            continue
        min_secs_since_last = MIN_SECS_SINCE_LAST_TRY.get(num_of_try, 3600 * 24)
        if time.mktime(email.created.timetuple()) > min_secs_since_last:
            email.status = post_office_models.STATUS.queued
            emails_to_update.append(email)
    result = {
        'new failed': len(create),
        'total failed': len(update) + len(create),
    }
    with transaction.atomic():
        notification_models.EmailRetryCount.objects.bulk_create(create)
        bulk_update(update, update_fields=['counter'])
        bulk_update(emails_to_update, update_fields=['status'])
    return result
