# coding: utf-8
import logging
from datetime import timedelta

import arrow
from django.conf import settings
from django.utils import timezone
from ids.exceptions import BackendError
from post_office import models as post_office_models
from post_office import mail

from review.lib import datetimes
from review.lib import lock
from review.notifications import retry
from review.notifications import creators
from review.notifications import mailing
from review.notifications.builders import build_notifications_for_personreview_changes

log = logging.getLogger(__name__)


@lock.get_lock_or_do_nothing_task
def create_notification_for_review_reminders():
    return len(creators.create_notifications_for_review_reminders())


@lock.get_lock_or_do_nothing_task
def create_notifications_for_person_review_changes():
    mails, processed = creators.create_notifications_for_person_review_changes()
    return len(mails)


@lock.get_lock_or_do_nothing_task(bind=True, max_retries=10)
def create_notifications_for_person_review_comments(self):
    try:
        mails, processed = creators.create_notifications_for_person_review_comments()
    except BackendError as exc:
        if self.request.retries == self.max_retries:
            raise
        else:
            raise self.retry(exc=exc, eta=timezone.now() + timedelta(seconds=60))
    if processed >= settings.MAX_OBJECTS_PER_TASK:
        create_notifications_for_person_review_comments.delay()
    return len(mails)


@lock.get_lock_or_do_nothing_task
def create_notifications_for_added_calibrators():
    return len(creators.create_notification_for_added_calibrators())


@lock.get_lock_or_do_nothing_task
def digest_about_changes_emailing():
    try:
        built = build_notifications_for_personreview_changes()
        log.info('notifications built %s', len(built['notifications']))
    except Exception:
        log.exception('Error trying to build notifications for digest')

    mailing.save_email_models(built['notifications'])
    log.info('Task digest_about_changes_emailing finished')


@lock.get_lock_or_do_nothing_task
def send_queued_emails():
    if settings.DISABLE_EMAILS:
        sent = 0
        failed = 'EMAILS DISABLED'
    else:
        sent, failed = mail.send_queued()
    return {
        'sent': sent,
        'failed': failed,
    }


@lock.get_lock_or_do_nothing_task
def retry_failed_emails():
    return retry.retry_failed()


@lock.get_lock_or_do_nothing_task
def remove_sent_emails():
    before_date = datetimes.shifted(
        arrow.now(),
        days=-settings.STORE_EMAILS_FOR_DAYS
    )
    return post_office_models.Email.objects.filter(
        status=post_office_models.STATUS.sent,
        last_updated__lte=before_date,
    ).delete()
